<?php
// guardar_publicacion.php
// Guarda publicación como JSON + imagen en /publicaciones

// ---------- Config/Helpers ----------
date_default_timezone_set('America/Argentina/Cordoba');

function base_url_dir() {
  // Carpeta donde está este archivo (p.ej. /generadoresargentina.com.ar/tienda)
  $b = rtrim(str_replace('\\','/', dirname($_SERVER['SCRIPT_NAME'] ?? '')), '/');
  return ($b === '/') ? '' : $b;
}
function redirect_ok()   { header('Location: ' . base_url_dir() . '/formulario.html?ok=1');   exit; }
function redirect_err($m){ header('Location: ' . base_url_dir() . '/formulario.html?error=1&msg=' . urlencode($m)); exit; }

function clean($s){ return trim((string)$s); }
function esc_name($s){
  $s = preg_replace('/[^a-zA-Z0-9\-\._]/','_', (string)$s);
  return substr($s, 0, 120);
}
function upload_err_text($code){
  return match ($code) {
    UPLOAD_ERR_INI_SIZE    => 'La imagen supera el tamaño permitido por el servidor.',
    UPLOAD_ERR_FORM_SIZE   => 'La imagen supera el tamaño máximo (5MB).',
    UPLOAD_ERR_PARTIAL     => 'La imagen se subió solo parcialmente.',
    UPLOAD_ERR_NO_FILE     => 'No se recibió ninguna imagen.',
    UPLOAD_ERR_NO_TMP_DIR  => 'Falta la carpeta temporal en el servidor.',
    UPLOAD_ERR_CANT_WRITE  => 'No se pudo escribir el archivo en disco.',
    UPLOAD_ERR_EXTENSION   => 'Una extensión de PHP detuvo la carga.',
    default                => 'Error desconocido al subir la imagen.',
  };
}

// ---------- Solo POST ----------
if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  redirect_err('Método no permitido.');
}

// ---------- Directorio de salida ----------
$dir = __DIR__ . '/publicaciones';
if (!is_dir($dir)) {
  if (!@mkdir($dir, 0755, true)) {
    redirect_err('No se pudo crear la carpeta de publicaciones.');
  }
}

// ---------- Campos ----------
$titulo    = isset($_POST['titulo'])    ? clean($_POST['titulo'])    : '';
$contenido = isset($_POST['contenido']) ? clean($_POST['contenido']) : '';
$link      = isset($_POST['link'])      ? clean($_POST['link'])      : '';

$sinPrecio = 1; // por defecto: sin precio
if (isset($_POST['sin_precio'])) {
  // viene "1" si sin precio, "0" si con precio
  $sinPrecio = ($_POST['sin_precio'] === '0') ? 0 : 1;
}

// Precio opcional
$precio = null;
if ($sinPrecio === 0 && isset($_POST['precio']) && $_POST['precio'] !== '') {
  $precio = (float)$_POST['precio'];
  if ($precio < 0) { $precio = 0; }
}

// ---------- Validaciones mínimas ----------
if ($titulo === '' || $contenido === '') {
  redirect_err('Título y descripción son obligatorios.');
}

// ---------- Imagen (requerida) ----------
if (!isset($_FILES['imagen'])) {
  redirect_err('No se recibió la imagen.');
}
if ($_FILES['imagen']['error'] !== UPLOAD_ERR_OK) {
  redirect_err(upload_err_text($_FILES['imagen']['error']));
}

// Tamaño máximo 5MB
if (!empty($_FILES['imagen']['size']) && $_FILES['imagen']['size'] > 5 * 1024 * 1024) {
  redirect_err('La imagen supera los 5MB.');
}

$imgTmp  = $_FILES['imagen']['tmp_name'];
$imgName = esc_name(basename($_FILES['imagen']['name'] ?? ''));
if ($imgName === '' || !is_uploaded_file($imgTmp)) {
  redirect_err('Archivo de imagen inválido.');
}

$ext = strtolower(pathinfo($imgName, PATHINFO_EXTENSION));
$permit = ['jpg','jpeg','png','webp'];
if (!in_array($ext, $permit, true)) {
  redirect_err('Formato de imagen no permitido. Usá JPG, PNG o WEBP.');
}

// Nombre único para la imagen
$imgOut = date('Ymd_His') . '_' . $imgName;
if (!@move_uploaded_file($imgTmp, $dir . '/' . $imgOut)) {
  redirect_err('No se pudo guardar la imagen.');
}

// ---------- Armar JSON ----------
$payload = [
  'titulo'    => $titulo,
  'contenido' => $contenido,
  'precio'    => $precio,   // null si es “sin precio”
  'imagen'    => $imgOut,
  'link'      => $link,
  'creado'    => date('c'),
];

// ---------- Guardar JSON ----------
$id   = uniqid('pub_', true);
$file = $dir . '/' . $id . '.json';
if (@file_put_contents($file, json_encode($payload, JSON_UNESCAPED_UNICODE|JSON_PRETTY_PRINT)) === false) {
  @unlink($dir . '/' . $imgOut);
  redirect_err('No se pudo guardar la publicación.');
}

// ---------- OK ----------
redirect_ok();
