<?php
// encuestas.php
require __DIR__ . '/db.php';
$pdo = pdo();

// Aseguramos la tabla (por si todavía no existe)
$pdo->exec("
  CREATE TABLE IF NOT EXISTS encuestas (
    id INT AUTO_INCREMENT PRIMARY KEY,
    titulo VARCHAR(255) NOT NULL,
    descripcion TEXT NULL,
    url VARCHAR(500) NULL,
    estado ENUM('borrador','publicada') NOT NULL DEFAULT 'publicada',
    created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP
  ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
");

// Traemos solo las encuestas publicadas
$stmt = $pdo->query("
  SELECT id, titulo, descripcion, url, created_at
  FROM encuestas
  WHERE estado = 'publicada'
  ORDER BY created_at DESC
");
$ENCUESTAS = $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
?>
<!DOCTYPE html>
<html lang="es">
<head>
  <meta charset="utf-8">
  <title>Encuestas · Fe.Tr.A</title>
  <?php require __DIR__ . '/_head_assets.php'; ?>
  <style>
    .encuestas-layout{
      padding:40px 0 60px;
    }
    .encuestas-header{
      margin-bottom:26px;
    }
    .encuestas-header h1{
      font-size:28px;
      margin:0 0 8px;
    }
    .encuestas-header p{
      margin:0;
      color:var(--muted);
      max-width:640px;
    }
    .encuestas-grid{
      display:grid;
      grid-template-columns:repeat(auto-fit,minmax(260px,1fr));
      gap:20px;
    }
    .encuesta-card{
      border-radius:14px;
      border:1px solid rgba(0,0,0,.06);
      box-shadow:0 10px 25px rgba(0,0,0,.04);
      padding:18px 18px 20px;
      background:#fff;
      display:flex;
      flex-direction:column;
      gap:10px;
    }
    .encuesta-card h2{
      font-size:18px;
      margin:0;
    }
    .encuesta-card p{
      margin:0;
      font-size:14px;
      color:var(--muted);
    }
    .encuesta-meta{
      font-size:12px;
      color:var(--muted);
      opacity:.9;
    }
    .btn-encuesta{
      margin-top:10px;
      align-self:flex-start;
      padding:8px 16px;
      border-radius:999px;
      border:none;
      background:var(--link);
      color:#fff;
      font-size:14px;
      font-weight:600;
      cursor:pointer;
      text-decoration:none;
    }
    .btn-encuesta:hover{
      filter:brightness(1.08);
    }
    .encuestas-empty{
      padding:30px 20px;
      border-radius:14px;
      background:rgba(21,50,76,.04);
      border:1px dashed rgba(21,50,76,.25);
      font-size:14px;
      color:var(--muted);
    }
  </style>
</head>
<body>

  <?php require __DIR__ . '/_navbar.php'; ?>

  <!-- Franja azul de fecha (debajo de la navbar) -->
  <div class="datebar">
    <div class="container inner">
      <span id="fecha-hoy">Hoy</span>
    </div>
  </div>

  <main>
    <section class="encuestas-layout">
      <div class="container">
        <header class="encuestas-header">
          <h1>Encuestas</h1>
          <p>
            Participá de las encuestas de la Federación y ayudanos a mejorar
            las condiciones del sector y los servicios para los afiliados.
          </p>
        </header>

        <?php if (!$ENCUESTAS): ?>
          <div class="encuestas-empty">
            Por el momento no hay encuestas publicadas. Vuelva a consultar más tarde.
          </div>
        <?php else: ?>
          <div class="encuestas-grid">
            <?php foreach ($ENCUESTAS as $e): ?>
              <article class="encuesta-card">
                <h2><?= htmlspecialchars($e['titulo'], ENT_QUOTES, 'UTF-8') ?></h2>

                <?php if (!empty($e['descripcion'])): ?>
                  <p><?= nl2br(htmlspecialchars($e['descripcion'], ENT_QUOTES, 'UTF-8')) ?></p>
                <?php endif; ?>

                <div class="encuesta-meta">
                  Publicada el <?= date('d/m/Y', strtotime($e['created_at'])) ?>
                </div>

                <a class="btn-encuesta"
                   href="ver_encuesta.php?id=<?= (int)$e['id'] ?>">
                  Ver encuesta
                </a>
              </article>
            <?php endforeach; ?>
          </div>
        <?php endif; ?>
      </div>
    </section>
  </main>

  <?php require __DIR__ . '/_footer.php'; ?>
  <?php require __DIR__ . '/_navbar_script.php'; ?>
</body>
</html>
