<?php
// admin_imagenes_notas.php
@session_start();

// Si tenés un sistema de login tipo "restringir.php", podés proteger así:
// require __DIR__.'/restringir.php';

$mensaje = '';
$errores = [];

// Rutas de las imágenes que usa el index
$imgDir   = __DIR__ . '/img/';
$webPath1 = 'img/imagen1.jpg';
$webPath2 = 'img/imagen2.jpg';
$file1    = $imgDir . 'imagen1.jpg';
$file2    = $imgDir . 'imagen2.jpg';

// Manejo de subida
if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    // helper simple para procesar cada input
    function procesar_imagen(string $campo, string $destino, array &$errores) {
        if (empty($_FILES[$campo]['name'])) {
            return false; // no se subió nada para este campo
        }

        if (!isset($_FILES[$campo]) || $_FILES[$campo]['error'] !== UPLOAD_ERR_OK) {
            $errores[] = "Error al subir el archivo de $campo.";
            return false;
        }

        $tmp  = $_FILES[$campo]['tmp_name'];
        $type = mime_content_type($tmp);

        // Aceptamos jpg / jpeg / png (se sobrescribe igual como .jpg)
        $permitidos = ['image/jpeg','image/jpg','image/png','image/webp'];
        if (!in_array($type, $permitidos, true)) {
            $errores[] = "El archivo de $campo debe ser una imagen (JPG, PNG o WEBP).";
            return false;
        }

        if (!is_uploaded_file($tmp)) {
            $errores[] = "El archivo de $campo no es válido.";
            return false;
        }

        if (!move_uploaded_file($tmp, $destino)) {
            $errores[] = "No se pudo guardar la imagen de $campo en el servidor.";
            return false;
        }

        return true;
    }

    $ok1 = procesar_imagen('imagen1', $file1, $errores);
    $ok2 = procesar_imagen('imagen2', $file2, $errores);

    if (!$errores) {
        if ($ok1 || $ok2) {
            $mensaje = 'Imágenes actualizadas correctamente.';
        } else {
            $mensaje = 'No se subió ningún archivo.';
        }
    }
}

// Para romper caché del navegador cuando se actualizan
$cachebuster = '?v=' . time();
?>
<!DOCTYPE html>
<html lang="es">
<head>
  <meta charset="utf-8">
  <title>Admin imágenes laterales de Notas</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <link rel="preconnect" href="https://fonts.googleapis.com">
  <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
  <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
  <style>
    :root {
      --primary: #4361ee;
      --primary-dark: #3a56d4;
      --secondary: #7209b7;
      --success: #06d6a0;
      --danger: #ef476f;
      --warning: #ffd166;
      --light: #f8f9fa;
      --dark: #212529;
      --gray-100: #f8f9fa;
      --gray-200: #e9ecef;
      --gray-300: #dee2e6;
      --gray-400: #ced4da;
      --gray-500: #adb5bd;
      --gray-600: #6c757d;
      --gray-700: #495057;
      --gray-800: #343a40;
      --gray-900: #212529;
      --shadow-sm: 0 1px 3px rgba(0,0,0,0.12), 0 1px 2px rgba(0,0,0,0.24);
      --shadow-md: 0 4px 6px rgba(0,0,0,0.1), 0 1px 3px rgba(0,0,0,0.08);
      --shadow-lg: 0 10px 25px rgba(0,0,0,0.1), 0 5px 10px rgba(0,0,0,0.05);
      --shadow-xl: 0 20px 40px rgba(0,0,0,0.15), 0 10px 10px rgba(0,0,0,0.05);
      --radius-sm: 4px;
      --radius: 8px;
      --radius-lg: 12px;
      --radius-xl: 16px;
    }

    * {
      margin: 0;
      padding: 0;
      box-sizing: border-box;
    }

    body {
      font-family: 'Inter', system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif;
      background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
      min-height: 100vh;
      color: var(--gray-800);
      line-height: 1.6;
    }

    .container {
      max-width: 1200px;
      margin: 0 auto;
      padding: 2rem 1rem;
    }

    .header {
      text-align: center;
      margin-bottom: 2.5rem;
    }

    .header h1 {
      font-size: 2.5rem;
      font-weight: 700;
      background: linear-gradient(90deg, var(--primary), var(--secondary));
      -webkit-background-clip: text;
      background-clip: text;
      color: transparent;
      margin-bottom: 0.5rem;
    }

    .header p {
      font-size: 1.1rem;
      color: var(--gray-600);
      max-width: 700px;
      margin: 0 auto;
    }

    .card-container {
      display: grid;
      grid-template-columns: repeat(auto-fit, minmax(450px, 1fr));
      gap: 2rem;
      margin-bottom: 2rem;
    }

    .card {
      background: white;
      border-radius: var(--radius-xl);
      box-shadow: var(--shadow-lg);
      overflow: hidden;
      transition: all 0.3s ease;
      border: 1px solid rgba(255, 255, 255, 0.2);
    }

    .card:hover {
      transform: translateY(-5px);
      box-shadow: var(--shadow-xl);
    }

    .card-header {
      padding: 1.5rem;
      background: linear-gradient(90deg, var(--primary), var(--secondary));
      color: white;
    }

    .card-header h2 {
      font-size: 1.25rem;
      font-weight: 600;
      margin: 0;
      display: flex;
      align-items: center;
      gap: 0.5rem;
    }

    .card-header h2::before {
      content: "";
      display: block;
      width: 8px;
      height: 8px;
      border-radius: 50%;
      background: white;
    }

    .card-body {
      padding: 1.5rem;
    }

    .preview-container {
      position: relative;
      border-radius: var(--radius-lg);
      overflow: hidden;
      margin-bottom: 1.5rem;
      background: var(--gray-100);
      border: 2px dashed var(--gray-300);
      min-height: 250px;
      display: flex;
      align-items: center;
      justify-content: center;
    }

    .preview-container.has-image {
      border: none;
    }

    .preview-image {
      width: 100%;
      height: auto;
      max-height: 300px;
      object-fit: cover;
      border-radius: var(--radius);
    }

    .no-image {
      text-align: center;
      color: var(--gray-500);
      padding: 2rem;
    }

    .no-image .icon {
      font-size: 3rem;
      margin-bottom: 0.5rem;
      opacity: 0.5;
    }

    .file-input-wrapper {
      position: relative;
      margin-bottom: 1rem;
    }

    .file-input {
      width: 100%;
      padding: 0.75rem 1rem;
      border: 1px solid var(--gray-300);
      border-radius: var(--radius);
      background: var(--gray-100);
      font-size: 0.9rem;
      transition: all 0.2s;
    }

    .file-input:focus {
      outline: none;
      border-color: var(--primary);
      box-shadow: 0 0 0 3px rgba(67, 97, 238, 0.15);
    }

    .file-input-label {
      display: block;
      margin-bottom: 0.5rem;
      font-weight: 500;
      color: var(--gray-700);
    }

    .help-text {
      font-size: 0.85rem;
      color: var(--gray-600);
      margin-top: 0.5rem;
    }

    .form-actions {
      text-align: center;
      margin-top: 2rem;
    }

    .btn {
      display: inline-flex;
      align-items: center;
      gap: 0.5rem;
      background: linear-gradient(90deg, var(--primary), var(--secondary));
      color: white;
      border: none;
      padding: 0.875rem 2rem;
      font-size: 1rem;
      font-weight: 600;
      border-radius: var(--radius-lg);
      cursor: pointer;
      transition: all 0.3s ease;
      box-shadow: var(--shadow-md);
    }

    .btn:hover {
      transform: translateY(-2px);
      box-shadow: var(--shadow-lg);
    }

    .btn:active {
      transform: translateY(0);
    }

    .btn-icon {
      font-size: 1.1rem;
    }

    .alert {
      padding: 1rem 1.5rem;
      border-radius: var(--radius-lg);
      margin-bottom: 2rem;
      box-shadow: var(--shadow-sm);
    }

    .alert-success {
      background: rgba(6, 214, 160, 0.1);
      border: 1px solid rgba(6, 214, 160, 0.3);
      color: #0a8754;
    }

    .alert-error {
      background: rgba(239, 71, 111, 0.1);
      border: 1px solid rgba(239, 71, 111, 0.3);
      color: #b91c1c;
    }

    .alert ul {
      margin: 0;
      padding-left: 1.5rem;
    }

    .info-box {
      background: white;
      border-radius: var(--radius-lg);
      padding: 1.5rem;
      margin-bottom: 2rem;
      box-shadow: var(--shadow-md);
      border-left: 4px solid var(--primary);
    }

    .info-box h3 {
      font-size: 1.1rem;
      margin-bottom: 0.5rem;
      color: var(--gray-800);
    }

    .info-box p {
      margin-bottom: 0.5rem;
      color: var(--gray-600);
    }

    .info-box code {
      background: var(--gray-100);
      padding: 0.2rem 0.4rem;
      border-radius: var(--radius-sm);
      font-size: 0.85rem;
      color: var(--primary);
    }

    @media (max-width: 768px) {
      .card-container {
        grid-template-columns: 1fr;
      }
      
      .header h1 {
        font-size: 2rem;
      }
      
      .container {
        padding: 1rem;
      }
    }

    /* Animaciones */
    @keyframes fadeIn {
      from { opacity: 0; transform: translateY(10px); }
      to { opacity: 1; transform: translateY(0); }
    }

    .fade-in {
      animation: fadeIn 0.5s ease forwards;
    }

    .stagger > * {
      opacity: 0;
      animation: fadeIn 0.5s ease forwards;
    }

    .stagger > *:nth-child(1) { animation-delay: 0.1s; }
    .stagger > *:nth-child(2) { animation-delay: 0.2s; }
    .stagger > *:nth-child(3) { animation-delay: 0.3s; }
  </style>
</head>
<body>
  <div class="container">
    <div class="header fade-in">
      <h1>Panel de Administración de Imágenes</h1>
      <p>Gestiona las imágenes laterales de la sección "Notas" en tu sitio web</p>
    </div>

    <div class="info-box fade-in">
      <h3>Información importante</h3>
      <p>Seleccioná nuevas imágenes y poné guardar, volvé al sitio y actuliza la página para ver los cambios. </p>
      <p>Formatos aceptados: JPG, PNG, WEBP</p>
    </div>

    <?php if ($mensaje): ?>
      <div class="alert alert-success fade-in">
        <?= htmlspecialchars($mensaje,ENT_QUOTES,'UTF-8') ?>
      </div>
    <?php endif; ?>

    <?php if ($errores): ?>
      <div class="alert alert-error fade-in">
        <ul>
          <?php foreach ($errores as $e): ?>
            <li><?= htmlspecialchars($e,ENT_QUOTES,'UTF-8') ?></li>
          <?php endforeach; ?>
        </ul>
      </div>
    <?php endif; ?>

    <form method="post" enctype="multipart/form-data" class="stagger">
      <div class="card-container">
        <div class="card">
          <div class="card-header">
            <h2>Imagen Izquierda</h2>
          </div>
          <div class="card-body">
            <div class="preview-container <?= is_file($file1) ? 'has-image' : '' ?>">
              <?php if (is_file($file1)): ?>
                <img src="<?= $webPath1 . $cachebuster ?>" alt="Imagen 1" class="preview-image">
              <?php else: ?>
                <div class="no-image">
                  <div class="icon">🖼️</div>
                  <p>No hay imagen1.jpg todavía</p>
                </div>
              <?php endif; ?>
            </div>
            <div class="file-input-wrapper">
              <label class="file-input-label">Seleccionar nueva imagen</label>
              <input type="file" name="imagen1" accept="image/*" class="file-input">
              <div class="help-text">Si subís algo acá, reemplaza la actual imagen1.jpg.</div>
            </div>
          </div>
        </div>

        <div class="card">
          <div class="card-header">
            <h2>Imagen Derecha</h2>
          </div>
          <div class="card-body">
            <div class="preview-container <?= is_file($file2) ? 'has-image' : '' ?>">
              <?php if (is_file($file2)): ?>
                <img src="<?= $webPath2 . $cachebuster ?>" alt="Imagen 2" class="preview-image">
              <?php else: ?>
                <div class="no-image">
                  <div class="icon">🖼️</div>
                  <p>No hay imagen2.jpg todavía</p>
                </div>
              <?php endif; ?>
            </div>
            <div class="file-input-wrapper">
              <label class="file-input-label">Seleccionar nueva imagen</label>
              <input type="file" name="imagen2" accept="image/*" class="file-input">
              <div class="help-text">Si subís algo acá, reemplaza la actual imagen2.jpg.</div>
            </div>
          </div>
        </div>
      </div>

      <div class="form-actions">
        <button class="btn" type="submit">
          <span class="btn-icon">💾</span>
          Guardar cambios
        </button>
      </div>
    </form>
  </div>

  <script>
    // Mejora visual: actualizar vista previa al seleccionar archivo
    document.querySelectorAll('input[type="file"]').forEach(input => {
      input.addEventListener('change', function() {
        const file = this.files[0];
        if (file) {
          const reader = new FileReader();
          const previewContainer = this.closest('.card-body').querySelector('.preview-container');
          
          reader.onload = function(e) {
            previewContainer.innerHTML = `<img src="${e.target.result}" alt="Vista previa" class="preview-image">`;
            previewContainer.classList.add('has-image');
          }
          
          reader.readAsDataURL(file);
        }
      });
    });
  </script>
</body>
</html>