<?php
// admin_encuestas.php
require __DIR__ . '/db.php';
$pdo = pdo();

/*
  Esquema de la tabla encuestas:

  - id
  - titulo
  - descripcion          (opcional)
  - pregunta             (obligatoria)
  - opciones_json        (array de opciones en JSON)
  - permite_comentario   (0/1)
  - estado               ('borrador'/'publicada')
  - created_at
*/

// ---- Crear tabla si no existe (ya con los campos nuevos) ----
$pdo->exec("
  CREATE TABLE IF NOT EXISTS encuestas (
    id INT AUTO_INCREMENT PRIMARY KEY,
    titulo VARCHAR(255) NOT NULL,
    descripcion TEXT NULL,
    pregunta TEXT NULL,
    opciones_json TEXT NULL,
    permite_comentario TINYINT(1) NOT NULL DEFAULT 1,
    estado ENUM('borrador','publicada') NOT NULL DEFAULT 'publicada',
    created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP
  ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
");

// ---- Asegurar columnas extra si la tabla ya existía con otro esquema ----
$colsStmt = $pdo->query("SHOW COLUMNS FROM encuestas");
$cols = $colsStmt ? $colsStmt->fetchAll(PDO::FETCH_COLUMN) : [];

if (!in_array('pregunta', $cols)) {
    $pdo->exec("ALTER TABLE encuestas ADD COLUMN pregunta TEXT NULL AFTER descripcion");
}
if (!in_array('opciones_json', $cols)) {
    $pdo->exec("ALTER TABLE encuestas ADD COLUMN opciones_json TEXT NULL AFTER pregunta");
}
if (!in_array('permite_comentario', $cols)) {
    $pdo->exec("ALTER TABLE encuestas ADD COLUMN permite_comentario TINYINT(1) NOT NULL DEFAULT 1 AFTER opciones_json");
}

// ----------------------------------------------------------------
//   LÓGICA CRUD
// ----------------------------------------------------------------
$errores = [];
$mensaje = null;

// Eliminar encuesta
if (isset($_GET['eliminar'])) {
    $id = (int)$_GET['eliminar'];
    if ($id > 0) {
        $stmt = $pdo->prepare("DELETE FROM encuestas WHERE id=? LIMIT 1");
        $stmt->execute([$id]);
        $mensaje = "Encuesta eliminada correctamente.";
    }
}

// Guardar / actualizar encuesta
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $id          = (int)($_POST['id'] ?? 0);
    $titulo      = trim($_POST['titulo'] ?? '');
    $descripcion = trim($_POST['descripcion'] ?? '');
    $pregunta    = trim($_POST['pregunta'] ?? '');
    $opcionesRaw = trim($_POST['opciones'] ?? '');
    $permite_comentario = isset($_POST['permite_comentario']) ? 1 : 0;
    $estado      = ($_POST['estado'] ?? 'publicada') === 'borrador' ? 'borrador' : 'publicada';

    // Validaciones: descripción NO es obligatoria
    if ($titulo === '') {
        $errores[] = "El título es obligatorio.";
    }
    if ($pregunta === '') {
        $errores[] = "La pregunta principal es obligatoria.";
    }

    // Procesar opciones (una por línea)
    $opciones_array = [];
    if ($opcionesRaw !== '') {
        $lines = preg_split('/\r\n|\r|\n/', $opcionesRaw);
        foreach ($lines as $line) {
            $line = trim($line);
            if ($line !== '') {
                $opciones_array[] = $line;
            }
        }
    }
    if (!$opciones_array) {
        $errores[] = "Debes definir al menos una opción de respuesta.";
    }

    $opciones_json = json_encode($opciones_array, JSON_UNESCAPED_UNICODE);

    if (!$errores) {
        if ($id > 0) {
            // UPDATE
            $stmt = $pdo->prepare("
                UPDATE encuestas
                SET titulo=?, descripcion=?, pregunta=?, opciones_json=?, permite_comentario=?, estado=?
                WHERE id=? LIMIT 1
            ");
            $stmt->execute([
                $titulo,
                $descripcion !== '' ? $descripcion : null,
                $pregunta,
                $opciones_json,
                $permite_comentario,
                $estado,
                $id
            ]);
            $mensaje = "Encuesta actualizada correctamente.";
        } else {
            // INSERT
            $stmt = $pdo->prepare("
                INSERT INTO encuestas (titulo, descripcion, pregunta, opciones_json, permite_comentario, estado)
                VALUES (?,?,?,?,?,?)
            ");
            $stmt->execute([
                $titulo,
                $descripcion !== '' ? $descripcion : null,
                $pregunta,
                $opciones_json,
                $permite_comentario,
                $estado
            ]);
            $mensaje = "Encuesta creada correctamente.";
        }
    }
}

// Modo edición
$edit = null;
$opciones_edit_text = '';
if (isset($_GET['editar'])) {
    $id = (int)$_GET['editar'];
    if ($id > 0) {
        $stmt = $pdo->prepare("SELECT * FROM encuestas WHERE id=? LIMIT 1");
        $stmt->execute([$id]);
        $edit = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($edit && !empty($edit['opciones_json'])) {
            $tmp = json_decode($edit['opciones_json'], true);
            if (is_array($tmp)) {
                $opciones_edit_text = implode("\n", $tmp);
            }
        }
    }
}

// Listado de encuestas
$stmt = $pdo->query("SELECT * FROM encuestas ORDER BY created_at DESC");
$LISTA = $stmt ? $stmt->fetchAll(PDO::FETCH_ASSOC) : [];
?>
<!DOCTYPE html>
<html lang="es">
<head>
  <meta charset="utf-8">
  <title>Admin encuestas · Fe.Tr.A</title>
  <?php require __DIR__ . '/_head_assets.php'; ?>
  <style>
    body{
      background:#f5f6f8;
      font-family:system-ui,-apple-system,BlinkMacSystemFont,"Segoe UI",sans-serif;
    }
    .admin-wrap{
      max-width:1100px;
      margin:30px auto 40px;
      padding:0 20px;
    }
    .admin-header{
      margin-bottom:18px;
    }
    .admin-header h1{
      font-size:24px;
      margin:0 0 4px;
    }
    .admin-header p{
      margin:0;
      color:var(--muted);
      font-size:14px;
    }

    .admin-grid{
      display:grid;
      grid-template-columns:minmax(0,360px) minmax(0,1fr);
      gap:24px;
      align-items:flex-start;
    }
    @media (max-width:900px){
      .admin-grid{ grid-template-columns:1fr; }
    }

    .card-admin{
      background:#fff;
      border-radius:16px;
      box-shadow:0 12px 30px rgba(15,26,36,.10);
      border:1px solid rgba(0,0,0,.04);
      padding:20px 20px 22px;
    }
    .card-admin h2{
      font-size:17px;
      margin:0 0 10px;
    }
    .form-row{
      margin-bottom:10px;
    }
    .form-row label{
      display:block;
      font-size:13px;
      font-weight:600;
      margin-bottom:4px;
    }
    .form-row textarea,
    .form-row input[type="text"],
    .form-row select{
      width:100%;
      border-radius:10px;
      border:1px solid rgba(0,0,0,.12);
      padding:7px 10px;
      font-size:14px;
      font-family:inherit;
    }
    .form-row textarea{
      min-height:80px;
      resize:vertical;
    }
    .form-row small{
      display:block;
      margin-top:2px;
      font-size:11px;
      color:var(--muted);
    }
    .form-check{
      display:flex;
      align-items:center;
      gap:6px;
      margin-bottom:8px;
      font-size:13px;
    }
    .btn-primary{
      border:none;
      border-radius:999px;
      background:var(--link);
      color:#fff;
      padding:8px 18px;
      font-size:14px;
      font-weight:600;
      cursor:pointer;
    }
    .btn-secondary{
      border:none;
      border-radius:999px;
      background:#e1e5ea;
      color:#333;
      padding:7px 14px;
      font-size:13px;
      cursor:pointer;
      text-decoration:none;
      display:inline-block;
      margin-left:6px;
    }
    .badge-estado{
      display:inline-flex;
      align-items:center;
      border-radius:999px;
      padding:3px 9px;
      font-size:11px;
      font-weight:600;
    }
    .badge-estado.publicada{
      background:rgba(46,204,113,.10);
      color:#27ae60;
    }
    .badge-estado.borrador{
      background:rgba(241,196,15,.12);
      color:#d68910;
    }
    .alert{
      padding:8px 12px;
      border-radius:10px;
      font-size:13px;
      margin-bottom:10px;
    }
    .alert-ok{
      background:rgba(46,204,113,.10);
      color:#27ae60;
      border:1px solid rgba(46,204,113,.35);
    }
    .alert-error{
      background:rgba(231,76,60,.10);
      color:#c0392b;
      border:1px solid rgba(231,76,60,.35);
    }
    table{
      width:100%;
      border-collapse:collapse;
      font-size:13px;
    }
    table th, table td{
      padding:6px 8px;
      border-bottom:1px solid rgba(0,0,0,.05);
      vertical-align:top;
    }
    table th{
      text-align:left;
      font-weight:600;
      font-size:12px;
      color:var(--muted);
    }
    table tr:last-child td{
      border-bottom:none;
    }
    .acciones a{
      font-size:12px;
      margin-right:8px;
      color:var(--link);
    }
    .acciones a.eliminar{
      color:#c0392b;
    }
    .texto-peq{
      font-size:12px;
      color:var(--muted);
    }
  </style>
</head>
<body>
  <div class="admin-wrap">
    <header class="admin-header">
      <h1>Administrar encuestas</h1>
      <p>Crea, edita y publica encuestas que luego se responden desde la sección pública <strong>Encuestas</strong>.</p>
    </header>

    <?php if ($mensaje): ?>
      <div class="alert alert-ok"><?= htmlspecialchars($mensaje, ENT_QUOTES, 'UTF-8') ?></div>
    <?php endif; ?>

    <?php if ($errores): ?>
      <div class="alert alert-error">
        <?php foreach ($errores as $e): ?>
          <div><?= htmlspecialchars($e, ENT_QUOTES, 'UTF-8') ?></div>
        <?php endforeach; ?>
      </div>
    <?php endif; ?>

    <div class="admin-grid">
      <!-- Formulario alta/edición -->
      <div class="card-admin">
        <h2><?= $edit ? 'Editar encuesta' : 'Nueva encuesta' ?></h2>
        <form method="post">
          <input type="hidden" name="id" value="<?= $edit ? (int)$edit['id'] : 0 ?>">

          <div class="form-row">
            <label for="titulo">Título</label>
            <input type="text" id="titulo" name="titulo"
                   value="<?= $edit ? htmlspecialchars($edit['titulo'], ENT_QUOTES, 'UTF-8') : '' ?>"
                   required>
          </div>

          <div class="form-row">
            <label for="descripcion">Descripción (visible en el listado)</label>
            <textarea id="descripcion" name="descripcion"><?= $edit ? htmlspecialchars($edit['descripcion'], ENT_QUOTES, 'UTF-8') : '' ?></textarea>
            <small>Texto corto explicando de qué trata la encuesta. Se ve en la card de <strong>encuestas.php</strong>. (Opcional)</small>
          </div>

          <div class="form-row">
            <label for="pregunta">Pregunta principal</label>
            <textarea id="pregunta" name="pregunta" required><?= $edit ? htmlspecialchars($edit['pregunta'], ENT_QUOTES, 'UTF-8') : '' ?></textarea>
            <small>Ejemplo: “¿Cómo evaluás las condiciones actuales de trabajo en el sector?”</small>
          </div>

          <div class="form-row">
            <label for="opciones">Opciones de respuesta (una por línea)</label>
            <textarea id="opciones" name="opciones" required><?= htmlspecialchars($opciones_edit_text, ENT_QUOTES, 'UTF-8') ?></textarea>
            <small>Ejemplo:<br>Muy buenas<br>Buenas<br>Regulares<br>Malas<br>Muy malas</small>
          </div>

          <div class="form-check">
            <input type="checkbox" id="permite_comentario" name="permite_comentario"
                   <?= $edit ? (!empty($edit['permite_comentario']) ? 'checked' : '') : 'checked' ?>>
            <label for="permite_comentario" style="margin:0;">Permitir comentario libre opcional</label>
          </div>

          <div class="form-row">
            <label for="estado">Estado</label>
            <?php $estSel = $edit ? $edit['estado'] : 'publicada'; ?>
            <select id="estado" name="estado">
              <option value="publicada" <?= $estSel === 'publicada' ? 'selected' : '' ?>>Publicada (visible en el sitio)</option>
              <option value="borrador"  <?= $estSel === 'borrador'  ? 'selected' : '' ?>>Borrador (oculta)</option>
            </select>
          </div>

          <button type="submit" class="btn-primary">
            <?= $edit ? 'Guardar cambios' : 'Crear encuesta' ?>
          </button>

          <?php if ($edit): ?>
            <a href="admin_encuestas.php" class="btn-secondary">Cancelar edición</a>
          <?php endif; ?>
        </form>
      </div>

      <!-- Listado encuestas -->
      <div class="card-admin">
        <h2>Listado de encuestas</h2>
        <?php if (!$LISTA): ?>
          <p class="texto-peq">Aún no hay encuestas cargadas.</p>
        <?php else: ?>
          <div style="overflow-x:auto;">
            <table>
              <thead>
                <tr>
                  <th>ID</th>
                  <th>Título</th>
                  <th>Pregunta</th>
                  <th>Estado</th>
                  <th>Fecha</th>
                  <th>Acciones</th>
                </tr>
              </thead>
              <tbody>
              <?php foreach ($LISTA as $row): ?>
                <tr>
                  <td><?= (int)$row['id'] ?></td>
                  <td><?= htmlspecialchars($row['titulo'], ENT_QUOTES, 'UTF-8') ?></td>
                  <td>
                    <?php
                      $preg = trim((string)$row['pregunta']);
                      if ($preg === '') {
                          echo '<span class="texto-peq">–</span>';
                      } else {
                          $short = mb_substr($preg, 0, 60, 'UTF-8');
                          if (mb_strlen($preg, 'UTF-8') > 60) $short .= '…';
                          echo htmlspecialchars($short, ENT_QUOTES, 'UTF-8');
                      }
                    ?>
                  </td>
                  <td>
                    <span class="badge-estado <?= $row['estado'] ?>">
                      <?= ucfirst($row['estado']) ?>
                    </span>
                  </td>
                  <td><?= date('d/m/Y', strtotime($row['created_at'])) ?></td>
                  <td class="acciones">
                    <a href="admin_encuesta_respuestas.php?id=<?= (int)$row['id'] ?>">Ver respuestas</a>
                    <a href="admin_encuestas.php?editar=<?= (int)$row['id'] ?>">Editar</a>
                    <a href="admin_encuestas.php?eliminar=<?= (int)$row['id'] ?>"
                       class="eliminar"
                       onclick="return confirm('¿Eliminar esta encuesta?');">
                      Eliminar
                    </a>
                  </td>
                </tr>
              <?php endforeach; ?>
              </tbody>
            </table>
          </div>
        <?php endif; ?>
      </div>
    </div>
  </div>
</body>
</html>
