<?php
// admin_encuesta_respuestas.php
require __DIR__ . '/db.php';
$pdo = pdo();

/*
 * Vista general de resultados de encuestas:
 * - Tabla con todas las encuestas + total de respuestas + con datos de contacto
 * - Si se selecciona una (GET id=...), abajo muestra:
 *     · Resumen con gráfico de barras por opción
 *     · Detalle de respuestas (comentario + nombre/email)
 */

// --------- Traer todas las encuestas ----------
$stmtE = $pdo->query("
    SELECT id, titulo, pregunta, created_at
    FROM encuestas
    ORDER BY created_at DESC
");
$encuestas = $stmtE ? $stmtE->fetchAll(PDO::FETCH_ASSOC) : [];

// Si no hay encuestas, no seguimos con más queries
$encuesta_id = (int)($_GET['id'] ?? 0);
if ($encuesta_id <= 0 && $encuestas) {
    // Por defecto seleccionamos la primera, solo para mostrar detalle
    $encuesta_id = (int)$encuestas[0]['id'];
}

// --------- Stats por encuesta: totales y con datos ---------
$stats = [];
$stmtStats = $pdo->query("
    SELECT encuesta_id,
           COUNT(*) AS total,
           SUM(
             CASE
               WHEN (nombre IS NOT NULL AND nombre <> '')
                 OR (apellido IS NOT NULL AND apellido <> '')
                 OR (email IS NOT NULL AND email <> '')
               THEN 1 ELSE 0
             END
           ) AS con_datos
    FROM encuestas_respuestas
    GROUP BY encuesta_id
");
if ($stmtStats) {
    foreach ($stmtStats->fetchAll(PDO::FETCH_ASSOC) as $row) {
        $stats[(int)$row['encuesta_id']] = [
            'total'     => (int)$row['total'],
            'con_datos' => (int)$row['con_datos'],
        ];
    }
}

// --------- Datos y respuestas de la encuesta seleccionada ----------
$E = null;
$resumen = [];
$detalle = [];
$totalRespuestasSeleccionada = 0;

if ($encuesta_id > 0) {
    // Traer datos de la encuesta
    $stmt = $pdo->prepare("
        SELECT id, titulo, descripcion, pregunta, created_at
        FROM encuestas
        WHERE id = ? LIMIT 1
    ");
    $stmt->execute([$encuesta_id]);
    $E = $stmt->fetch(PDO::FETCH_ASSOC);

    if ($E) {
        // Resumen por opción
        $stmt = $pdo->prepare("
            SELECT opcion, COUNT(*) as total
            FROM encuestas_respuestas
            WHERE encuesta_id = ?
            GROUP BY opcion
            ORDER BY total DESC
        ");
        $stmt->execute([$encuesta_id]);
        $resumen = $stmt->fetchAll(PDO::FETCH_ASSOC);

        foreach ($resumen as $r) {
            $totalRespuestasSeleccionada += (int)$r['total'];
        }

        // Detalle respuestas
        $stmt = $pdo->prepare("
            SELECT opcion, comentario, nombre, apellido, email, created_at
            FROM encuestas_respuestas
            WHERE encuesta_id = ?
            ORDER BY created_at DESC
        ");
        $stmt->execute([$encuesta_id]);
        $detalle = $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
  <meta charset="utf-8">
  <title>Respuestas encuestas · Fe.Tr.A</title>
  <?php require __DIR__ . '/_head_assets.php'; ?>
  <style>
    body{
      background:#f5f6f8;
      font-family:system-ui,-apple-system,BlinkMacSystemFont,"Segoe UI",sans-serif;
    }
    .admin-wrap{
      max-width:1150px;
      margin:30px auto 40px;
      padding:0 20px;
    }
    .admin-header h1{
      font-size:24px;
      margin:0 0 4px;
    }
    .admin-header p{
      margin:0;
      color:var(--muted);
      font-size:14px;
    }
    .admin-header small{
      font-size:12px;
      color:var(--muted);
    }
    .back-link{
      display:inline-flex;
      align-items:center;
      gap:6px;
      font-size:13px;
      text-decoration:none;
      color:var(--link);
      margin-bottom:10px;
    }
    .back-link span{
      font-size:16px;
      line-height:0;
    }
    .card{
      background:#fff;
      border-radius:16px;
      box-shadow:0 10px 24px rgba(15,26,36,.10);
      border:1px solid rgba(0,0,0,.04);
      padding:18px 18px 20px;
      margin-top:16px;
    }
    .card h2{
      font-size:17px;
      margin:0 0 10px;
    }
    .card h3{
      font-size:15px;
      margin:0 0 8px;
    }
    .texto-peq{
      font-size:12px;
      color:var(--muted);
    }
    table{
      width:100%;
      border-collapse:collapse;
      font-size:13px;
    }
    table th, table td{
      padding:6px 8px;
      border-bottom:1px solid rgba(0,0,0,.05);
      vertical-align:top;
    }
    table th{
      text-align:left;
      font-weight:600;
      font-size:12px;
      color:var(--muted);
    }
    table tr:last-child td{
      border-bottom:none;
    }
    .badge-opcion{
      display:inline-flex;
      padding:2px 8px;
      border-radius:999px;
      background:rgba(21,50,76,.06);
      font-size:11px;
    }
    .badge-estado{
      display:inline-flex;
      padding:2px 7px;
      border-radius:999px;
      font-size:11px;
      background:rgba(21,50,76,.06);
    }
    .badge-contacto{
      display:inline-flex;
      padding:2px 7px;
      border-radius:999px;
      font-size:11px;
      background:rgba(46,204,113,.10);
      color:#27ae60;
    }
    .encuesta-row-activa{
      background:rgba(21,50,76,.03);
    }
    .grid-detalle{
      display:grid;
      grid-template-columns:minmax(0,340px) minmax(0,1fr);
      gap:20px;
      margin-top:20px;
    }
    @media (max-width:900px){
      .grid-detalle{ grid-template-columns:1fr; }
    }
    .resumen-row{
      display:flex;
      justify-content:space-between;
      align-items:center;
      margin-bottom:6px;
      font-size:13px;
    }
    .resumen-row strong{
      font-weight:600;
    }
    .resumen-bar{
      height:6px;
      border-radius:999px;
      background:rgba(33,150,243,.12);
      overflow:hidden;
      margin-top:3px;
    }
    .resumen-bar span{
      display:block;
      height:100%;
      background:var(--link);
    }
    .acciones a{
      font-size:12px;
      margin-right:6px;
      color:var(--link);
      text-decoration:none;
    }
  </style>
</head>
<body>
  <div class="admin-wrap">

    <a href="admin/dashboard_fetra.php" class="back-link">
      <span>←</span> Volver al Dashboard
    </a>

    <header class="admin-header">
      <h1>Respuestas de encuestas</h1>
      <p>Resumen general de participación y detalle de respuestas por encuesta.</p>
    </header>

    <!-- LISTADO DE TODAS LAS ENCUESTAS + TOTALES -->
    <div class="card">
      <h2>Encuestas y cantidad de respuestas</h2>

      <?php if (!$encuestas): ?>
        <p class="texto-peq">Todavía no hay encuestas creadas.</p>
      <?php else: ?>
        <div style="overflow-x:auto;">
          <table>
            <thead>
              <tr>
                <th>ID</th>
                <th>Título</th>
                <th>Pregunta</th>
                <th>Respuestas</th>
                <th>Con datos de contacto</th>
                <th>Acciones</th>
              </tr>
            </thead>
            <tbody>
            <?php foreach ($encuestas as $row): 
              $id = (int)$row['id'];
              $st = $stats[$id] ?? ['total' => 0, 'con_datos' => 0];
            ?>
              <tr class="<?= $id === $encuesta_id ? 'encuesta-row-activa' : '' ?>">
                <td><?= $id ?></td>
                <td><?= htmlspecialchars($row['titulo'], ENT_QUOTES, 'UTF-8') ?></td>
                <td>
                  <?php
                    $preg = trim((string)$row['pregunta']);
                    if ($preg === '') {
                        echo '<span class="texto-peq">–</span>';
                    } else {
                        $short = mb_substr($preg, 0, 60, 'UTF-8');
                        if (mb_strlen($preg, 'UTF-8') > 60) $short .= '…';
                        echo htmlspecialchars($short, ENT_QUOTES, 'UTF-8');
                    }
                  ?>
                </td>
                <td>
                  <?php if ($st['total'] > 0): ?>
                    <strong><?= $st['total'] ?></strong>
                  <?php else: ?>
                    <span class="texto-peq">0</span>
                  <?php endif; ?>
                </td>
                <td>
                  <?php if ($st['con_datos'] > 0): ?>
                    <span class="badge-contacto">
                      <?= $st['con_datos'] ?> con datos
                    </span>
                  <?php else: ?>
                    <span class="texto-peq">Sin datos de contacto</span>
                  <?php endif; ?>
                </td>
                <td class="acciones">
                  <a href="admin_encuesta_respuestas.php?id=<?= $id ?>">Ver detalle</a>
                </td>
              </tr>
            <?php endforeach; ?>
            </tbody>
          </table>
        </div>
      <?php endif; ?>
    </div>

    <?php if ($E): ?>
      <!-- DETALLE DE LA ENCUESTA SELECCIONADA -->
      <div class="grid-detalle">

        <!-- RESUMEN + GRÁFICO -->
        <div class="card">
          <h2>Resumen encuesta seleccionada</h2>
          <p><strong><?= htmlspecialchars($E['titulo'], ENT_QUOTES, 'UTF-8') ?></strong></p>
          <?php if (!empty($E['pregunta'])): ?>
            <p class="texto-peq"><?= htmlspecialchars($E['pregunta'], ENT_QUOTES, 'UTF-8') ?></p>
          <?php endif; ?>

          <p class="texto-peq">
            Total de respuestas: <strong><?= (int)$totalRespuestasSeleccionada ?></strong>
          </p>

          <?php if ($totalRespuestasSeleccionada == 0): ?>
            <p class="texto-peq">Todavía no hay respuestas registradas para esta encuesta.</p>
          <?php else: ?>
            <?php foreach ($resumen as $r):
              $cnt = (int)$r['total'];
              $pct = $totalRespuestasSeleccionada > 0 ? round($cnt * 100 / $totalRespuestasSeleccionada, 1) : 0;
            ?>
              <div class="resumen-row">
                <span><?= htmlspecialchars($r['opcion'], ENT_QUOTES, 'UTF-8') ?></span>
                <span><strong><?= $cnt ?></strong> (<?= $pct ?>%)</span>
              </div>
              <div class="resumen-bar">
                <span style="width: <?= $pct ?>%;"></span>
              </div>
            <?php endforeach; ?>
          <?php endif; ?>
        </div>

        <!-- DETALLE RESPUESTAS -->
        <div class="card">
          <h2>Detalle de respuestas</h2>

          <?php if (!$detalle): ?>
            <p class="texto-peq">No hay respuestas para esta encuesta.</p>
          <?php else: ?>
            <div style="overflow-x:auto;">
              <table>
                <thead>
                  <tr>
                    <th>Fecha</th>
                    <th>Opción elegida</th>
                    <th>Comentario</th>
                    <th>Nombre</th>
                    <th>Email</th>
                  </tr>
                </thead>
                <tbody>
                <?php foreach ($detalle as $row): ?>
                  <tr>
                    <td><?= date('d/m/Y H:i', strtotime($row['created_at'])) ?></td>
                    <td>
                      <span class="badge-opcion">
                        <?= htmlspecialchars($row['opcion'], ENT_QUOTES, 'UTF-8') ?>
                      </span>
                    </td>
                    <td>
                      <?php if (!empty($row['comentario'])): ?>
                        <?= nl2br(htmlspecialchars($row['comentario'], ENT_QUOTES, 'UTF-8')) ?>
                      <?php else: ?>
                        <span class="texto-peq">–</span>
                      <?php endif; ?>
                    </td>
                    <td>
                      <?php
                        $nombreCompleto = trim(($row['nombre'] ?? '') . ' ' . ($row['apellido'] ?? ''));
                        echo $nombreCompleto !== ''
                          ? htmlspecialchars($nombreCompleto, ENT_QUOTES, 'UTF-8')
                          : '<span class="texto-peq">Anónimo</span>';
                      ?>
                    </td>
                    <td>
                      <?php if (!empty($row['email'])): ?>
                        <a href="mailto:<?= htmlspecialchars($row['email'], ENT_QUOTES, 'UTF-8') ?>">
                          <?= htmlspecialchars($row['email'], ENT_QUOTES, 'UTF-8') ?>
                        </a>
                      <?php else: ?>
                        <span class="texto-peq">–</span>
                      <?php endif; ?>
                    </td>
                  </tr>
                <?php endforeach; ?>
                </tbody>
              </table>
            </div>
          <?php endif; ?>
        </div>

      </div>
    <?php endif; ?>

  </div>
</body>
</html>
