<?php 
// cliente/operacion.php
@session_start();
if (empty($_SESSION['cliente_id'])) {
    header('Location: login.php');
    exit;
}

require_once '../db.php';
$pdo = pdo();

function h($v){ return htmlspecialchars((string)$v, ENT_QUOTES, 'UTF-8'); }

$cliente_id   = (int)$_SESSION['cliente_id'];
$operacion_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;

if ($operacion_id <= 0) {
    die('Operación no especificada.');
}

/* =========================
   0) Mensajes flash (para evitar repost en F5)
   ========================= */
$alert_ok  = '';
$alert_err = [];

if (!empty($_SESSION['flash_ok'])) {
    $alert_ok = $_SESSION['flash_ok'];
    unset($_SESSION['flash_ok']);
}
if (!empty($_SESSION['flash_err']) && is_array($_SESSION['flash_err'])) {
    $alert_err = $_SESSION['flash_err'];
    unset($_SESSION['flash_err']);
}

/* =========================
   1) Traer operación del cliente
   ========================= */
$stmt = $pdo->prepare("
    SELECT o.*, c.nombre, c.apellido, c.dni
    FROM operaciones o
    INNER JOIN clientes c ON c.id = o.cliente_id
    WHERE o.id = :op_id AND c.id = :cli_id
    LIMIT 1
");
$stmt->execute([
    ':op_id'  => $operacion_id,
    ':cli_id' => $cliente_id
]);
$op = $stmt->fetch();

if (!$op) {
    die('No se encontró la operación para este cliente.');
}

/* =========================
   2) Procesar alta de pago SUBIDO POR CLIENTE
   (con POST-Redirect-GET)
   ========================= */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['registrar_pago_cliente'])) {

    $errores_local = [];

    $medio        = $_POST['medio'] ?? '';
    $monto        = (float)($_POST['monto'] ?? 0);
    $comentario   = trim($_POST['comentario'] ?? '');
    $cuota_numero = isset($_POST['cuota_numero_cliente'])
        ? (int)$_POST['cuota_numero_cliente']
        : 0;

    $medios_validos = ['transferencia','pago_facil','rapipago'];
    if (!in_array($medio, $medios_validos, true)) {
        $errores_local[] = 'Medio de pago inválido.';
    }
    if ($monto <= 0) {
        $errores_local[] = 'El monto debe ser mayor a 0.';
    }
    if ($cuota_numero <= 0) {
        $errores_local[] = 'Debés indicar qué cuota estás pagando.';
    }

    // Manejo de archivo (opcional)
    $ruta_final = null;
    if (!empty($_FILES['comprobante']['name'])) {
        $file = $_FILES['comprobante'];
        if ($file['error'] === UPLOAD_ERR_OK) {

            $permitidos = ['image/jpeg','image/png','application/pdf'];
            if (!in_array($file['type'], $permitidos, true)) {
                $errores_local[] = 'El comprobante debe ser JPG, PNG o PDF.';
            } else {
                $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));

                $dir = __DIR__ . '/../uploads/comprobantes/';
                if (!is_dir($dir)) {
                    @mkdir($dir, 0775, true);
                }

                $nombre_archivo = 'op'.$operacion_id.'_cli'.$cliente_id.'_'.time().'.'.$ext;
                $destino        = $dir . $nombre_archivo;

                if (!@move_uploaded_file($file['tmp_name'], $destino)) {
                    $errores_local[] = 'No se pudo guardar el comprobante en el servidor.';
                } else {
                    $ruta_final = 'uploads/comprobantes/'.$nombre_archivo;
                }
            }
        } elseif ($file['error'] !== UPLOAD_ERR_NO_FILE) {
            $errores_local[] = 'Error al subir el archivo de comprobante.';
        }
    }

    if (empty($errores_local)) {
        try {
            $stmtIns = $pdo->prepare("
                INSERT INTO pagos_solicitados
                    (operacion_id, cliente_id, medio, monto, cuota_numero, comprobante_path, comentario, estado, fecha_solicitud)
                VALUES
                    (:op_id, :cli_id, :medio, :monto, :cuota_numero, :path, :comentario, 'pendiente', NOW())
            ");
            $stmtIns->execute([
                ':op_id'        => $operacion_id,
                ':cli_id'       => $cliente_id,
                ':medio'        => $medio,
                ':monto'        => $monto,
                ':cuota_numero' => $cuota_numero,
                ':path'         => $ruta_final,
                ':comentario'   => $comentario ?: null,
            ]);

            // Guardamos mensaje en sesión y redirigimos (PRG)
            $_SESSION['flash_ok'] = 'Tu comprobante fue enviado. Un asesor lo revisará y confirmará el pago.';
            header('Location: operacion.php?id='.$operacion_id);
            exit;

        } catch (Exception $e) {
            $errores_local[] = 'Ocurrió un error al registrar el pago: '.$e->getMessage();
        }
    }

    // Si hubo errores en este POST, los pasamos a flash y redirigimos igual
    // (así tampoco se reenvía con F5; el usuario ve los errores en GET)
    if (!empty($errores_local)) {
        $_SESSION['flash_err'] = $errores_local;
        header('Location: operacion.php?id='.$operacion_id);
        exit;
    }
}

/* =========================
   3) Traer pagos confirmados (tabla pagos)
   ========================= */
$stmt = $pdo->prepare("
    SELECT *
    FROM pagos
    WHERE operacion_id = :op_id
    ORDER BY fecha_pago ASC, id ASC
");
$stmt->execute([':op_id' => $operacion_id]);
$pagos = $stmt->fetchAll();

/* =========================
   4) Traer pagos solicitados por el cliente
   ========================= */
$stmt = $pdo->prepare("
    SELECT *
    FROM pagos_solicitados
    WHERE operacion_id = :op_id AND cliente_id = :cli_id
    ORDER BY fecha_solicitud DESC
");
$stmt->execute([
    ':op_id'  => $operacion_id,
    ':cli_id' => $cliente_id
]);
$pagos_solic = $stmt->fetchAll();

// Pendientes por cuota (para marcar en el plan)
$pendientes_por_cuota = [];
foreach ($pagos_solic as $s) {
    if (!empty($s['cuota_numero']) && $s['estado'] === 'pendiente') {
        $n = (int)$s['cuota_numero'];
        $pendientes_por_cuota[$n] = ($pendientes_por_cuota[$n] ?? 0) + (float)$s['monto'];
    }
}

/* =========================
   5) Cálculos generales
   ========================= */
$precio_lista = (float)$op['precio_lista'];

$senia           = (float)$op['senia'];
$entVehiculo     = (float)$op['entrega_vehiculo'];
$entTerreno      = (float)$op['entrega_terreno'];
$entCheques      = (float)$op['entrega_cheques'];

$entregas_iniciales = $senia + $entVehiculo + $entTerreno + $entCheques;

// pagos confirmados (sobre saldo)
$total_pagos = 0.0;
foreach ($pagos as $p) {
    $total_pagos += (float)$p['monto'];
}

$total_abonado = $entregas_iniciales + $total_pagos;
$saldo_total   = $precio_lista - $total_abonado;
if ($saldo_total < 0) $saldo_total = 0;

$saldo_financiar = (float)$op['saldo_financiar'];
$saldo_pend_fin  = $saldo_financiar - $total_pagos;
if ($saldo_pend_fin < 0) $saldo_pend_fin = 0;

$porcentaje = ($precio_lista > 0)
    ? round(($total_abonado / $precio_lista) * 100)
    : 0;
if ($porcentaje > 100) $porcentaje = 100;

/* =========================
   6) Plan de cuotas (operaciones_cuotas o teórico)
   ========================= */
$cuotas_plan     = [];
$cuotas_plan_ok  = false;

try {
    $stmt = $pdo->prepare("
        SELECT *
        FROM operaciones_cuotas
        WHERE operacion_id = :op_id
        ORDER BY numero ASC
    ");
    $stmt->execute([':op_id' => $operacion_id]);
    $cuotas_plan = $stmt->fetchAll();
    $cuotas_plan_ok = !empty($cuotas_plan);
} catch (Exception $e) {
    $cuotas_plan     = [];
    $cuotas_plan_ok  = false;
}

$cuotas_resumen = [];
$proxima_cuota  = null;

$cuotas_total = (int)$op['cuotas_cantidad'];
$cuota_monto  = (float)$op['cuota_monto'];

if ($cuotas_plan_ok) {
    foreach ($cuotas_plan as $fila) {
        $n      = (int)$fila['numero'];
        $monto  = (float)$fila['monto'];
        $pagado = (float)$fila['pagado'];
        $estado = (string)$fila['estado'];

        $cuotas_resumen[$n] = [
            'numero' => $n,
            'monto'  => $monto,
            'pagado' => $pagado,
            'estado' => $estado,
        ];
    }

    foreach ($cuotas_resumen as $n => $info) {
        if ($info['estado'] !== 'pagada') {
            $proxima_cuota = $n;
            break;
        }
    }
} elseif ($cuotas_total > 0 && $cuota_monto > 0) {
    for ($n = 1; $n <= $cuotas_total; $n++) {
        $cuotas_resumen[$n] = [
            'numero' => $n,
            'monto'  => $cuota_monto,
            'pagado' => 0.0,
            'estado' => 'pendiente',
        ];
    }

    foreach ($pagos as $p) {
        $tipo = $p['tipo'] ?? 'cuota';
        if ($tipo === 'cuota' && !empty($p['cuota_numero'])) {
            $n = (int)$p['cuota_numero'];
            if (isset($cuotas_resumen[$n])) {
                $cuotas_resumen[$n]['pagado'] += (float)$p['monto'];
            }
        }
    }

    foreach ($cuotas_resumen as $n => &$info) {
        if ($info['pagado'] >= ($info['monto'] - 0.01)) {
            $info['estado'] = 'pagada';
        } elseif ($info['pagado'] > 0) {
            $info['estado'] = 'parcial';
        } else {
            $info['estado'] = 'pendiente';
        }
    }
    unset($info);

    foreach ($cuotas_resumen as $n => $info) {
        if ($info['estado'] === 'pendiente') {
            $proxima_cuota = $n;
            break;
        }
    }
}

function label_estado_cuota($estado){
    switch ($estado) {
        case 'pagada':   return 'Pagada';
        case 'parcial':  return 'Pagada parcialmente';
        case 'pendiente':
        default:         return 'Pendiente';
    }
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <title>Detalle de operación <?php echo h($op['codigo_operacion']); ?> - Cabañas Santiago</title>
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
</head>
<body class="bg-light">
<nav class="navbar navbar-expand-lg navbar-dark bg-primary">
    <div class="container-fluid">
        <a class="navbar-brand" href="panel.php">Cabañas Santiago - Clientes</a>
        <div class="d-flex">
            <span class="navbar-text me-3" style="font-size:0.9rem;">
                <?php echo h($op['nombre'] . ' ' . $op['apellido']); ?>
                (DNI: <?php echo h($op['dni']); ?>)
            </span>
            <a class="btn btn-outline-light btn-sm" href="logout.php">Salir</a>
        </div>
    </div>
</nav>

<div class="container py-4">

    <a href="panel.php" class="btn btn-link btn-sm mb-3">&laquo; Volver a mis operaciones</a>

    <h3 class="mb-2">Operación <?php echo h($op['codigo_operacion']); ?></h3>
    <p class="text-muted mb-3">
        Fecha de alta: <?php echo h($op['fecha_alta']); ?>
    </p>

    <?php if ($alert_ok): ?>
        <div class="alert alert-success"><?php echo h($alert_ok); ?></div>
    <?php endif; ?>
    <?php if (!empty($alert_err)): ?>
        <div class="alert alert-danger">
            <ul class="mb-0">
                <?php foreach ($alert_err as $e): ?>
                    <li><?php echo h($e); ?></li>
                <?php endforeach; ?>
            </ul>
        </div>
    <?php endif; ?>

    <div class="row g-3 mb-4">
        <!-- Detalle vivienda -->
        <div class="col-md-6">
            <div class="card shadow-sm h-100">
                <div class="card-body">
                    <h5 class="card-title">Detalle de la vivienda</h5>
                    <p class="mb-1">
                        <strong>Modelo:</strong> <?php echo h($op['modelo']); ?>
                    </p>
                    <p class="mb-1">
                        <strong>Metros cubiertos:</strong>
                        <?php echo (float)$op['metros_cubiertos']; ?> m²
                    </p>
                    <p class="mb-1">
                        <strong>Metros de galería:</strong>
                        <?php echo (float)$op['metros_galeria']; ?> m²
                    </p>
                    <p class="mb-1">
                        <strong>Sistema constructivo:</strong><br>
                        <?php echo h($op['sistema']); ?>
                    </p>
                    <p class="mb-1">
                        <strong>Forma de venta:</strong>
                        <?php echo h($op['forma_venta']); ?>
                    </p>
                    <p class="mb-1">
                        <strong>Tipo de plan:</strong>
                        <?php echo h($op['tipo_plan']); ?>
                    </p>
                    <p class="mb-1">
                        <strong>Estado de la obra:</strong>
                        <?php echo h($op['estado_obra']); ?>
                    </p>
                    <p class="mb-1">
                        <strong>Fecha estimada de entrega:</strong>
                        <?php echo $op['fecha_estimada_entrega']
                            ? h($op['fecha_estimada_entrega'])
                            : 'A definir'; ?>
                    </p>
                    <p class="mb-1">
                        <strong>Fecha de entrega real:</strong>
                        <?php echo $op['fecha_entrega_real']
                            ? h($op['fecha_entrega_real'])
                            : '-'; ?>
                    </p>
                    <?php if (!empty($op['observaciones'])): ?>
                        <p class="mt-2 mb-0">
                            <strong>Observaciones:</strong><br>
                            <?php echo nl2br(h($op['observaciones'])); ?>
                        </p>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <!-- Estado pagos -->
        <div class="col-md-6">
            <div class="card shadow-sm h-100">
                <div class="card-body">
                    <h5 class="card-title">Estado de pagos</h5>

                    <p class="mb-1">
                        <strong>Precio total de la vivienda:</strong><br>
                        $ <?php echo number_format($precio_lista, 2, ',', '.'); ?>
                    </p>

                    <hr>

                    <p class="mb-1">
                        <strong>Entregas iniciales:</strong><br>
                        $ <?php echo number_format($entregas_iniciales, 2, ',', '.'); ?>
                    </p>
                    <ul class="small text-muted">
                        <li>Seña / efectivo: $ <?php echo number_format($senia, 2, ',', '.'); ?></li>
                        <li>Vehículo tomado: $ <?php echo number_format($entVehiculo, 2, ',', '.'); ?></li>
                        <li>Terreno tomado: $ <?php echo number_format($entTerreno, 2, ',', '.'); ?></li>
                        <li>Cheques: $ <?php echo number_format($entCheques, 2, ',', '.'); ?></li>
                    </ul>

                    <p class="mb-1">
                        <strong>Saldo a financiar:</strong><br>
                        $ <?php echo number_format($saldo_financiar, 2, ',', '.'); ?>
                    </p>
                    <p class="mb-1">
                        <strong>Pagos confirmados sobre saldo:</strong><br>
                        $ <?php echo number_format($total_pagos, 2, ',', '.'); ?>
                    </p>
                    <p class="mb-1">
                        <strong>Saldo pendiente del plan:</strong><br>
                        <span class="fs-5 text-success">
                            $ <?php echo number_format($saldo_pend_fin, 2, ',', '.'); ?>
                        </span>
                    </p>

                    <hr>

                    <p class="mb-1">
                        <strong>Total abonado (entregas + pagos):</strong><br>
                        $ <?php echo number_format($total_abonado, 2, ',', '.'); ?>
                    </p>
                    <p class="mb-3">
                        <strong>Saldo total pendiente sobre la vivienda:</strong><br>
                        <span class="fs-5 text-success">
                            $ <?php echo number_format($saldo_total, 2, ',', '.'); ?>
                        </span>
                    </p>

                    <div class="mb-3">
                        <div class="progress" style="height: 10px;">
                            <div class="progress-bar" role="progressbar"
                                 style="width: <?php echo $porcentaje; ?>%;"
                                 aria-valuenow="<?php echo $porcentaje; ?>"
                                 aria-valuemin="0" aria-valuemax="100"></div>
                        </div>
                        <small class="text-muted">
                            Avance de pagos: <?php echo $porcentaje; ?>%
                        </small>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Plan de cuotas -->
    <?php if (!empty($cuotas_resumen)): ?>
        <h5 class="mb-2">Plan de cuotas</h5>
        <p class="text-muted">
            <?php if ($cuotas_plan_ok): ?>
                Plan de cuotas generado por Cabañas Santiago. Los montos pueden ser distintos por cuota.
            <?php else: ?>
                Plan teórico de <?php echo (int)$cuotas_total; ?> cuotas de
                $ <?php echo number_format($cuota_monto, 2, ',', '.'); ?> cada una.
            <?php endif; ?>
            <?php if ($proxima_cuota !== null): ?>
                <br><strong>Próxima cuota sugerida:</strong> N° <?php echo (int)$proxima_cuota; ?>
            <?php endif; ?>
        </p>

        <div class="table-responsive mb-4">
            <table class="table table-sm table-striped align-middle">
                <thead class="table-light">
                <tr>
                    <th>Cuota</th>
                    <th>Monto plan</th>
                    <th>Pagado</th>
                    <th>Estado</th>
                </tr>
                </thead>
                <tbody>
                <?php foreach ($cuotas_resumen as $info): ?>
                    <?php
                    $n              = (int)$info['numero'];
                    $is_next        = ($proxima_cuota !== null && $n == $proxima_cuota);
                    $pendiente_conf = isset($pendientes_por_cuota[$n]);
                    $estado_label   = label_estado_cuota($info['estado']);
                    if ($pendiente_conf) {
                        $estado_label .= ' — pago pendiente de confirmación';
                    }
                    ?>
                    <tr class="<?php echo $is_next ? 'table-warning' : ''; ?>">
                        <td><?php echo $n; ?></td>
                        <td>$ <?php echo number_format($info['monto'], 2, ',', '.'); ?></td>
                        <td>$ <?php echo number_format($info['pagado'], 2, ',', '.'); ?></td>
                        <td><?php echo h($estado_label); ?></td>
                    </tr>
                <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    <?php endif; ?>

    <!-- Formulario: registrar pago con comprobante -->
    <div class="card mb-4 shadow-sm">
        <div class="card-body">
            <h5 class="card-title">Registrar un pago con comprobante</h5>
            <p class="text-muted">
                Usá este formulario solo si abonaste mediante
                <strong>transferencia</strong> o <strong>Pago Fácil / Rapipago</strong>.
                Un asesor verificará el comprobante y luego se impactará en tu plan de cuotas.
            </p>

            <form method="post" enctype="multipart/form-data" class="row g-3">
                <input type="hidden" name="registrar_pago_cliente" value="1">

                <div class="col-md-4">
                    <label class="form-label">Medio de pago *</label>
                    <select name="medio" class="form-select" required>
                        <option value="">Seleccionar...</option>
                        <option value="transferencia">Transferencia bancaria</option>
                        <option value="pago_facil">Pago Fácil</option>
                        <option value="rapipago">Rapipago</option>
                    </select>
                </div>

                <div class="col-md-4">
                    <label class="form-label">Cuota que estás pagando *</label>
                    <select name="cuota_numero_cliente" class="form-select" required>
                        <option value="">Seleccionar...</option>
                        <?php foreach ($cuotas_resumen as $info): ?>
                            <?php if ($info['estado'] !== 'pagada'): ?>
                                <?php
                                $n  = (int)$info['numero'];
                                $sel = ($proxima_cuota !== null && $n == $proxima_cuota) ? 'selected' : '';
                                $lbl_estado = label_estado_cuota($info['estado']);
                                ?>
                                <option value="<?php echo $n; ?>" <?php echo $sel; ?>>
                                    Cuota <?php echo $n; ?> — $ <?php echo number_format($info['monto'],2,',','.'); ?>
                                    (<?php echo $lbl_estado; ?>)
                                </option>
                            <?php endif; ?>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="col-md-4">
                    <label class="form-label">Monto abonado *</label>
                    <input type="number" step="0.01" min="0" name="monto"
                           class="form-control" required>
                </div>

                <div class="col-md-6">
                    <label class="form-label">Comprobante (JPG/PNG/PDF)</label>
                    <input type="file" name="comprobante" class="form-control"
                           accept=".jpg,.jpeg,.png,.pdf">
                </div>

                <div class="col-12">
                    <label class="form-label">Comentarios (opcional)</label>
                    <textarea name="comentario" class="form-control" rows="2"
                              placeholder="Ej: Fecha y hora del pago, últimos 4 dígitos de la cuenta, etc."></textarea>
                </div>

                <div class="col-12 text-end">
                    <button type="submit" class="btn btn-primary">
                        Enviar comprobante
                    </button>
                </div>
            </form>
        </div>
    </div>

    <!-- Solicitudes de pago del cliente -->
    <h5 class="mb-3">Comprobantes enviados</h5>
    <?php if (empty($pagos_solic)): ?>
        <div class="alert alert-info">
            Aún no registraste pagos con comprobante desde este portal.
        </div>
    <?php else: ?>
        <div class="table-responsive mb-4">
            <table class="table table-sm table-striped align-middle">
                <thead class="table-light">
                <tr>
                    <th>Fecha envío</th>
                    <th>Medio</th>
                    <th>Cuota</th>
                    <th>Monto</th>
                    <th>Estado</th>
                    <th>Comprobante</th>
                    <th>Comentario</th>
                </tr>
                </thead>
                <tbody>
                <?php foreach ($pagos_solic as $s): ?>
                    <tr>
                        <td><?php echo h($s['fecha_solicitud']); ?></td>
                        <td><?php echo h($s['medio']); ?></td>
                        <td><?php echo $s['cuota_numero'] ? (int)$s['cuota_numero'] : '-'; ?></td>
                        <td>$ <?php echo number_format($s['monto'], 2, ',', '.'); ?></td>
                        <td>
                            <?php
                            if ($s['estado'] === 'aprobado') {
                                echo '<span class="badge bg-success">Aprobado</span>';
                            } elseif ($s['estado'] === 'rechazado') {
                                echo '<span class="badge bg-danger">Rechazado</span>';
                            } else {
                                echo '<span class="badge bg-warning text-dark">Pendiente</span>';
                            }
                            ?>
                        </td>
                        <td>
                            <?php if (!empty($s['comprobante_path'])): ?>
                                <a href="../<?php echo h($s['comprobante_path']); ?>" target="_blank"
                                   class="btn btn-sm btn-outline-secondary">
                                    Ver comprobante
                                </a>
                            <?php else: ?>
                                -
                            <?php endif; ?>
                        </td>
                        <td><?php echo nl2br(h($s['comentario'] ?? '')); ?></td>
                    </tr>
                <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    <?php endif; ?>

    <!-- Historial de pagos confirmados -->
    <h5 class="mb-3">Historial de pagos confirmados</h5>

    <?php if (empty($pagos)): ?>
        <div class="alert alert-info">
            Por el momento no se registran pagos confirmados adicionales a las entregas iniciales.
        </div>
    <?php else: ?>
        <div class="table-responsive">
            <table class="table table-sm table-striped align-middle">
                <thead class="table-light">
                <tr>
                    <th>Fecha</th>
                    <th>Tipo</th>
                    <th>Cuota</th>
                    <th>Monto</th>
                    <th>Medio de pago</th>
                    <th>Referencia</th>
                    <th>Detalle</th>
                </tr>
                </thead>
                <tbody>
                <?php foreach ($pagos as $p): ?>
                    <tr>
                        <td><?php echo h($p['fecha_pago']); ?></td>
                        <td><?php echo h($p['tipo'] ?? 'cuota'); ?></td>
                        <td>
                            <?php
                            $tipo = $p['tipo'] ?? 'cuota';
                            echo $tipo === 'cuota' && !empty($p['cuota_numero'])
                                ? (int)$p['cuota_numero']
                                : '-';
                            ?>
                        </td>
                        <td>$ <?php echo number_format($p['monto'], 2, ',', '.'); ?></td>
                        <td><?php echo h($p['medio']); ?></td>
                        <td><?php echo h($p['referencia'] ?? ''); ?></td>
                        <td><?php echo nl2br(h($p['observaciones'] ?? '')); ?></td>
                    </tr>
                <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    <?php endif; ?>

</div>
</body>
</html>
