<?php 
// cliente/operacion.php
@session_start();
if (empty($_SESSION['cliente_id'])) {
    header('Location: login.php');
    exit;
}

require_once '../db.php';
$pdo = pdo();

function h($v){ return htmlspecialchars((string)$v, ENT_QUOTES, 'UTF-8'); }

$cliente_id   = (int)$_SESSION['cliente_id'];
$operacion_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;

if ($operacion_id <= 0) {
    die('Operación no especificada.');
}

/* =========================
   1) Traer operación del cliente
   ========================= */
$stmt = $pdo->prepare("
    SELECT o.*, c.nombre, c.apellido, c.dni
    FROM operaciones o
    INNER JOIN clientes c ON c.id = o.cliente_id
    WHERE o.id = :op_id AND c.id = :cli_id
    LIMIT 1
");
$stmt->execute([
    ':op_id'  => $operacion_id,
    ':cli_id' => $cliente_id
]);
$op = $stmt->fetch();

if (!$op) {
    die('No se encontró la operación para este cliente.');
}

/* =========================
   2) Traer pagos
   ========================= */
$stmt = $pdo->prepare("
    SELECT *
    FROM pagos
    WHERE operacion_id = :op_id
    ORDER BY fecha_pago ASC, id ASC
");
$stmt->execute([':op_id' => $operacion_id]);
$pagos = $stmt->fetchAll();

/* =========================
   3) Traer plan de cuotas (si existe operaciones_cuotas)
   ========================= */
$cuotas_plan = [];
$cuotas_plan_sql_ok = false;

try {
    $stmt = $pdo->prepare("
        SELECT *
        FROM operaciones_cuotas
        WHERE operacion_id = :op_id
        ORDER BY numero ASC
    ");
    $stmt->execute([':op_id' => $operacion_id]);
    $cuotas_plan = $stmt->fetchAll();
    $cuotas_plan_sql_ok = true;
} catch (Exception $e) {
    // Si la tabla no existe aún, simplemente no usamos plan explícito
    $cuotas_plan = [];
    $cuotas_plan_sql_ok = false;
}

/* =========================
   4) Cálculos generales
   ========================= */
$precio_lista = (float)$op['precio_lista'];

// entregas iniciales
$senia            = (float)$op['senia'];
$entregaVehiculo  = (float)$op['entrega_vehiculo'];
$entregaTerreno   = (float)$op['entrega_terreno'];
$entregaCheques   = (float)$op['entrega_cheques'];

$entregas_iniciales = $senia + $entregaVehiculo + $entregaTerreno + $entregaCheques;

// pagos sobre saldo (cuotas + extras)
$total_pagos = 0.0;
foreach ($pagos as $p) {
    $total_pagos += (float)$p['monto'];
}

$total_abonado = $entregas_iniciales + $total_pagos;
$saldo_total   = $precio_lista - $total_abonado;
if ($saldo_total < 0) $saldo_total = 0;

// saldo a financiar de la operación (precio_lista - entregas iniciales)
$saldo_financiar = (float)$op['saldo_financiar'];

// saldo pendiente de ese saldo a financiar
$saldo_pendiente_financiar = $saldo_financiar - $total_pagos;
if ($saldo_pendiente_financiar < 0) $saldo_pendiente_financiar = 0;

// porcentaje de avance general sobre precio de lista
$porcentaje = ($precio_lista > 0)
    ? round(($total_abonado / $precio_lista) * 100)
    : 0;
if ($porcentaje > 100) $porcentaje = 100;

/* =========================
   5) Estado de cuotas
   ========================= */
$cuotas_resumen = [];   // array con info por cuota para mostrar
$proxima_cuota  = null; // número de próxima cuota pendiente

// datos de operación
$cuotas_total = (int)$op['cuotas_cantidad'];
$cuota_monto  = (float)$op['cuota_monto'];

/**
 * Caso A: existe tabla operaciones_cuotas con filas para esta operación
 *         → usamos esos datos como plan “real”.
 */
if ($cuotas_plan_sql_ok && !empty($cuotas_plan)) {

    foreach ($cuotas_plan as $fila) {
        $n       = (int)$fila['numero'];
        $monto   = (float)$fila['monto'];
        $pagado  = (float)$fila['pagado'];
        $estado  = (string)$fila['estado'];

        $cuotas_resumen[$n] = [
            'numero' => $n,
            'monto'  => $monto,
            'pagado' => $pagado,
            'estado' => $estado,
        ];
    }

    // próxima cuota: primera que no esté marcada como "pagada"
    foreach ($cuotas_resumen as $n => $info) {
        if ($info['estado'] !== 'pagada') {
            $proxima_cuota = $n;
            break;
        }
    }

/**
 * Caso B: no hay plan guardado → calculamos en base a cuotas_cantidad y pagos
 */
} elseif ($cuotas_total > 0 && $cuota_monto > 0) {

    // inicializar cuotas “teóricas”
    for ($n = 1; $n <= $cuotas_total; $n++) {
        $cuotas_resumen[$n] = [
            'numero' => $n,
            'monto'  => $cuota_monto,
            'pagado' => 0.0,
            'estado' => 'pendiente',
        ];
    }

    // acumular pagos tipo cuota por número de cuota
    foreach ($pagos as $p) {
        $tipo = $p['tipo'] ?? 'cuota';
        if ($tipo === 'cuota' && !empty($p['cuota_numero'])) {
            $n = (int)$p['cuota_numero'];
            if (isset($cuotas_resumen[$n])) {
                $cuotas_resumen[$n]['pagado'] += (float)$p['monto'];
            }
        }
    }

    // determinar estado según monto plan
    foreach ($cuotas_resumen as $n => &$info) {
        if ($info['pagado'] >= ($info['monto'] - 0.01)) {
            $info['estado'] = 'pagada';
        } elseif ($info['pagado'] > 0) {
            $info['estado'] = 'parcial';
        } else {
            $info['estado'] = 'pendiente';
        }
    }
    unset($info);

    // próxima cuota = primera pendiente
    foreach ($cuotas_resumen as $n => $info) {
        if ($info['estado'] === 'pendiente') {
            $proxima_cuota = $n;
            break;
        }
    }
}

/* helper para texto estado cuota */
function label_estado_cuota($estado){
    switch ($estado) {
        case 'pagada':   return 'Pagada';
        case 'parcial':  return 'Pagada parcialmente';
        case 'pendiente':
        default:         return 'Pendiente';
    }
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <title>Detalle de operación <?php echo h($op['codigo_operacion']); ?> - Cabañas Santiago</title>
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
</head>
<body class="bg-light">
<nav class="navbar navbar-expand-lg navbar-dark bg-primary">
    <div class="container-fluid">
        <a class="navbar-brand" href="panel.php">Cabañas Santiago - Clientes</a>
        <div class="d-flex">
            <span class="navbar-text me-3" style="font-size:0.9rem;">
                <?php echo h($op['nombre'] . ' ' . $op['apellido']); ?>
                (DNI: <?php echo h($op['dni']); ?>)
            </span>
            <a class="btn btn-outline-light btn-sm" href="logout.php">Salir</a>
        </div>
    </div>
</nav>

<div class="container py-4">

    <a href="panel.php" class="btn btn-link btn-sm mb-3">&laquo; Volver a mis operaciones</a>

    <h3 class="mb-2">Operación <?php echo h($op['codigo_operacion']); ?></h3>
    <p class="text-muted mb-3">
        Fecha de alta: <?php echo h($op['fecha_alta']); ?>
    </p>

    <div class="row g-3 mb-4">
        <!-- Detalle vivienda -->
        <div class="col-md-6">
            <div class="card shadow-sm h-100">
                <div class="card-body">
                    <h5 class="card-title">Detalle de la vivienda</h5>
                    <p class="mb-1">
                        <strong>Modelo:</strong> <?php echo h($op['modelo']); ?>
                    </p>
                    <p class="mb-1">
                        <strong>Metros cubiertos:</strong>
                        <?php echo (float)$op['metros_cubiertos']; ?> m²
                    </p>
                    <p class="mb-1">
                        <strong>Metros de galería:</strong>
                        <?php echo (float)$op['metros_galeria']; ?> m²
                    </p>
                    <p class="mb-1">
                        <strong>Sistema constructivo:</strong><br>
                        <?php echo h($op['sistema']); ?>
                    </p>
                    <p class="mb-1">
                        <strong>Forma de venta:</strong>
                        <?php echo h($op['forma_venta']); ?>
                    </p>
                    <p class="mb-1">
                        <strong>Tipo de plan:</strong>
                        <?php echo h($op['tipo_plan']); ?>
                        <?php if ($op['tipo_plan'] === 'plan_ahorro'): ?>
                            <br><small class="text-muted">
                                Inicio de obra al <?php echo (float)$op['porcentaje_inicio_obra']; ?>% del valor abonado.
                            </small>
                        <?php endif; ?>
                    </p>
                    <p class="mb-1">
                        <strong>Estado de la obra:</strong>
                        <?php echo h($op['estado_obra']); ?>
                    </p>
                    <p class="mb-1">
                        <strong>Fecha estimada de entrega:</strong>
                        <?php echo $op['fecha_estimada_entrega']
                            ? h($op['fecha_estimada_entrega'])
                            : 'A definir'; ?>
                    </p>
                    <p class="mb-1">
                        <strong>Fecha de entrega real:</strong>
                        <?php echo $op['fecha_entrega_real']
                            ? h($op['fecha_entrega_real'])
                            : '-'; ?>
                    </p>
                    <?php if (!empty($op['observaciones'])): ?>
                        <p class="mt-2 mb-0">
                            <strong>Observaciones:</strong><br>
                            <?php echo nl2br(h($op['observaciones'])); ?>
                        </p>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <!-- Estado pagos -->
        <div class="col-md-6">
            <div class="card shadow-sm h-100">
                <div class="card-body">
                    <h5 class="card-title">Estado de pagos</h5>

                    <p class="mb-1">
                        <strong>Precio total de la vivienda:</strong><br>
                        $ <?php echo number_format($precio_lista, 2, ',', '.'); ?>
                    </p>

                    <hr>

                    <p class="mb-1">
                        <strong>Entregas iniciales:</strong><br>
                        $ <?php echo number_format($entregas_iniciales, 2, ',', '.'); ?>
                    </p>
                    <ul class="small text-muted">
                        <li>Seña / efectivo: $ <?php echo number_format($senia, 2, ',', '.'); ?></li>
                        <li>Vehículo tomado: $ <?php echo number_format($entregaVehiculo, 2, ',', '.'); ?></li>
                        <li>Terreno tomado: $ <?php echo number_format($entregaTerreno, 2, ',', '.'); ?></li>
                        <li>Cheques: $ <?php echo number_format($entregaCheques, 2, ',', '.'); ?></li>
                    </ul>

                    <p class="mb-1">
                        <strong>Saldo a financiar:</strong><br>
                        $ <?php echo number_format($saldo_financiar, 2, ',', '.'); ?>
                    </p>
                    <p class="mb-1">
                        <strong>Pagos registrados sobre saldo:</strong><br>
                        $ <?php echo number_format($total_pagos, 2, ',', '.'); ?>
                    </p>
                    <p class="mb-1">
                        <strong>Saldo pendiente del plan:</strong><br>
                        <span class="fs-5 text-success">
                            $ <?php echo number_format($saldo_pendiente_financiar, 2, ',', '.'); ?>
                        </span>
                    </p>

                    <hr>

                    <p class="mb-1">
                        <strong>Total abonado (entregas + pagos):</strong><br>
                        $ <?php echo number_format($total_abonado, 2, ',', '.'); ?>
                    </p>
                    <p class="mb-3">
                        <strong>Saldo total pendiente sobre la vivienda:</strong><br>
                        <span class="fs-5 text-success">
                            $ <?php echo number_format($saldo_total, 2, ',', '.'); ?>
                        </span>
                    </p>

                    <div class="mb-3">
                        <div class="progress" style="height: 10px;">
                            <div class="progress-bar" role="progressbar"
                                 style="width: <?php echo $porcentaje; ?>%;"
                                 aria-valuenow="<?php echo $porcentaje; ?>"
                                 aria-valuemin="0" aria-valuemax="100"></div>
                        </div>
                        <small class="text-muted">
                            Avance de pagos: <?php echo $porcentaje; ?>%
                        </small>
                    </div>

                    <h6 class="mt-3">Formas de pago</h6>
                    <ul class="mb-0">
                        <li>Transferencia bancaria</li>
                        <li>Tarjetas de débito y crédito</li>
                        <li>Pago Fácil / Rapipago</li>
                        <li>MercadoPago (consultar datos)</li>
                    </ul>
                    <small class="text-muted">
                        Para realizar un pago o informar un comprobante,
                        comunicate con Cabañas Santiago por los canales habituales.
                    </small>
                </div>
            </div>
        </div>
    </div>

    <!-- Plan de cuotas -->
    <?php if (!empty($cuotas_resumen)): ?>
        <h5 class="mb-2">Plan de cuotas</h5>
        <p class="text-muted">
            <?php if ($cuotas_plan_sql_ok && !empty($cuotas_plan)): ?>
                Plan generado por Cabañas Santiago. Las cuotas pueden tener montos distintos.
            <?php else: ?>
                Plan teórico de <?php echo (int)$cuotas_total; ?> cuotas de
                $ <?php echo number_format($cuota_monto, 2, ',', '.'); ?> cada una.
            <?php endif; ?>
            <?php if ($proxima_cuota !== null): ?>
                <br><strong>Próxima cuota a pagar:</strong> N° <?php echo (int)$proxima_cuota; ?>
            <?php endif; ?>
        </p>

        <div class="table-responsive mb-4">
            <table class="table table-sm table-striped align-middle">
                <thead class="table-light">
                <tr>
                    <th>Cuota</th>
                    <th>Monto plan</th>
                    <th>Pagado</th>
                    <th>Estado</th>
                </tr>
                </thead>
                <tbody>
                <?php foreach ($cuotas_resumen as $info): ?>
                    <?php
                    $is_next = ($proxima_cuota !== null && $info['numero'] == $proxima_cuota);
                    ?>
                    <tr class="<?php echo $is_next ? 'table-warning' : ''; ?>">
                        <td><?php echo (int)$info['numero']; ?></td>
                        <td>$ <?php echo number_format($info['monto'], 2, ',', '.'); ?></td>
                        <td>$ <?php echo number_format($info['pagado'], 2, ',', '.'); ?></td>
                        <td><?php echo h(label_estado_cuota($info['estado'])); ?></td>
                    </tr>
                <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    <?php endif; ?>

    <!-- Historial de pagos -->
    <h5 class="mb-3">Historial de pagos</h5>

    <?php if (empty($pagos)): ?>
        <div class="alert alert-info">
            Por el momento no se registran pagos adicionales a las entregas iniciales.
        </div>
    <?php else: ?>
        <div class="table-responsive">
            <table class="table table-sm table-striped align-middle">
                <thead class="table-light">
                <tr>
                    <th>Fecha</th>
                    <th>Tipo</th>
                    <th>Cuota</th>
                    <th>Monto</th>
                    <th>Medio de pago</th>
                    <th>Referencia</th>
                    <th>Detalle</th>
                </tr>
                </thead>
                <tbody>
                <?php foreach ($pagos as $p): ?>
                    <tr>
                        <td><?php echo h($p['fecha_pago']); ?></td>
                        <td><?php echo h($p['tipo'] ?? 'cuota'); ?></td>
                        <td>
                            <?php
                            $tipo = $p['tipo'] ?? 'cuota';
                            echo $tipo === 'cuota' && !empty($p['cuota_numero'])
                                ? (int)$p['cuota_numero']
                                : '-';
                            ?>
                        </td>
                        <td>$ <?php echo number_format($p['monto'], 2, ',', '.'); ?></td>
                        <td><?php echo h($p['medio']); ?></td>
                        <td><?php echo h($p['referencia'] ?? ''); ?></td>
                        <td><?php echo nl2br(h($p['observaciones'] ?? '')); ?></td>
                    </tr>
                <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    <?php endif; ?>

</div>
</body>
</html>
