<?php
@session_start();
require_once 'db.php';
$pdo = pdo();

/* =========================
   1) Traer modelo elegido
   ========================= */
$modelo_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;

$stmt = $pdo->prepare("SELECT * FROM modelos_casas WHERE id = :id AND activo = 1 LIMIT 1");
$stmt->execute([':id' => $modelo_id]);
$modelo = $stmt->fetch();

if (!$modelo) {
    http_response_code(404);
    echo "Modelo no encontrado.";
    exit;
}

/* =========================
   Helpers
   ========================= */
function limpiar($v) {
    return trim($v ?? '');
}

function generar_codigo_acceso() {
    return strtoupper(substr(bin2hex(random_bytes(4)), 0, 8)); // 8 caracteres
}

$errores = [];
$ok = false;
$nuevo_codigo = null;

/* =========================
   2) Procesar envío
   ========================= */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    // Datos personales
    $nombre    = limpiar($_POST['nombre']    ?? '');
    $apellido  = limpiar($_POST['apellido']  ?? '');
    $dni       = limpiar($_POST['dni']       ?? '');
    $email     = filter_var($_POST['email'] ?? '', FILTER_SANITIZE_EMAIL);
    $telefono  = limpiar($_POST['telefono']  ?? '');
    $localidad = limpiar($_POST['localidad'] ?? '');

    // Sistema y forma de pago deseada
    $sistema = limpiar($_POST['sistema'] ?? $modelo['sistema_default']);
    if (!in_array($sistema, ['estructura','estructura_plataforma','llave_en_mano'], true)) {
        $sistema = $modelo['sistema_default'];
    }

    $forma_venta_deseada = limpiar($_POST['forma_venta_deseada'] ?? '');
    if (!in_array($forma_venta_deseada, ['contado','financiado','mixto'], true)) {
        $errores[] = "Debés elegir cómo te gustaría pagar.";
    }

    // Montos que SI tiene sentido que el cliente diga
    $senia_deseada   = (float)str_replace(',', '.', $_POST['senia_deseada']   ?? 0);
    $cuotas_deseadas = (int)($_POST['cuotas_deseadas'] ?? 0);

    // Vehículo / Terreno / Cheques → SOLO detalle, el monto lo define el admin
    $entrega_vehiculo_detalle = limpiar($_POST['entrega_vehiculo_detalle'] ?? '');
    $entrega_terreno_detalle  = limpiar($_POST['entrega_terreno_detalle']  ?? '');
    $entrega_cheques_detalle  = limpiar($_POST['entrega_cheques_detalle']  ?? '');

    // Comentarios del cliente (tu campo real es comentarios_cliente)
    $comentarios_cliente = limpiar($_POST['observaciones'] ?? '');
    if ($entrega_cheques_detalle !== '') {
        if ($comentarios_cliente !== '') {
            $comentarios_cliente .= "\n\nDetalle de cheques: " . $entrega_cheques_detalle;
        } else {
            $comentarios_cliente = "Detalle de cheques: " . $entrega_cheques_detalle;
        }
    }

    // Validaciones básicas
    if ($nombre === '' || $apellido === '' || $dni === '' || $telefono === '') {
        $errores[] = "Completá nombre, apellido, DNI y teléfono.";
    }
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $errores[] = "El email no parece válido.";
    }

    if (empty($errores)) {
        try {
            $pdo->beginTransaction();

            /* =========================
               2.1) Buscar o crear cliente
               ========================= */
            $cliente_id = null;

            $stmt = $pdo->prepare("SELECT * FROM clientes WHERE dni = :dni LIMIT 1");
            $stmt->execute([':dni' => $dni]);
            $cliente = $stmt->fetch();

            if ($cliente) {
                $cliente_id = (int)$cliente['id'];

                // OJO: si tu tabla clientes no tiene updated_at, sacá esa columna del UPDATE
                $upd = $pdo->prepare("
                    UPDATE clientes
                    SET nombre = :nombre,
                        apellido = :apellido,
                        email = :email,
                        telefono = :telefono,
                        localidad = :localidad,
                        updated_at = NOW()
                    WHERE id = :id
                ");
                $upd->execute([
                    ':nombre'    => $nombre,
                    ':apellido'  => $apellido,
                    ':email'     => $email,
                    ':telefono'  => $telefono,
                    ':localidad' => $localidad,
                    ':id'        => $cliente_id
                ]);

                $nuevo_codigo = $cliente['codigo_acceso'];
            } else {
                $codigo = generar_codigo_acceso();

                // OJO: si tu tabla clientes no tiene created_at, sacá esa columna del INSERT
                $ins = $pdo->prepare("
                    INSERT INTO clientes
                        (nombre, apellido, dni, email, telefono, localidad, codigo_acceso, created_at)
                    VALUES
                        (:nombre, :apellido, :dni, :email, :telefono, :localidad, :codigo, NOW())
                ");
                $ins->execute([
                    ':nombre'    => $nombre,
                    ':apellido'  => $apellido,
                    ':dni'       => $dni,
                    ':email'     => $email,
                    ':telefono'  => $telefono,
                    ':localidad' => $localidad,
                    ':codigo'    => $codigo
                ]);

                $cliente_id   = (int)$pdo->lastInsertId();
                $nuevo_codigo = $codigo;
            }

            /* =========================
               2.2) Insertar simulación
               ========================= */

            // Precio de referencia: usamos precio_lista si está > 0, sino precio_desde
            $precio_ref = (float)$modelo['precio_lista'];
            if ($precio_ref <= 0) {
                $precio_ref = (float)$modelo['precio_desde'];
            }

            // Montos de vehículo/terreno/cheques → 0 en simulación (los define admin)
            $entrega_vehiculo_deseada = 0;
            $entrega_terreno_deseada  = 0;
            $entrega_cheques_deseada  = 0;

            $insSim = $pdo->prepare("
                INSERT INTO simulaciones_operacion
                    (cliente_id,
                     operacion_id,
                     modelo,
                     metros_cubiertos,
                     metros_galeria,
                     sistema,
                     precio_referencia,
                     total_final,
                     forma_venta_deseada,
                     senia_deseada,
                     entrega_vehiculo_deseada,
                     entrega_vehiculo_detalle,
                     entrega_terreno_deseada,
                     entrega_terreno_detalle,
                     entrega_cheques_deseada,
                     cuotas_deseadas,
                     cuota_monto_estimado,
                     comentarios_cliente,
                     estado,
                     created_at)
                VALUES
                    (:cliente_id,
                     NULL,
                     :modelo,
                     :m_cubiertos,
                     :m_galeria,
                     :sistema,
                     :precio_ref,
                     NULL,
                     :forma_venta,
                     :senia,
                     :ent_veh,
                     :det_veh,
                     :ent_terr,
                     :det_terr,
                     :ent_cheq,
                     :cuotas,
                     0,
                     :comentarios,
                     'pendiente',
                     NOW())
            ");

            $insSim->execute([
                ':cliente_id'  => $cliente_id,
                ':modelo'      => $modelo['nombre'],
                ':m_cubiertos' => $modelo['metros_cubiertos'],
                ':m_galeria'   => $modelo['metros_galeria'],
                ':sistema'     => $sistema,
                ':precio_ref'  => $precio_ref,
                ':forma_venta' => $forma_venta_deseada,
                ':senia'       => $senia_deseada,
                ':ent_veh'     => $entrega_vehiculo_deseada,
                ':det_veh'     => $entrega_vehiculo_detalle,
                ':ent_terr'    => $entrega_terreno_deseada,
                ':det_terr'    => $entrega_terreno_detalle,
                ':ent_cheq'    => $entrega_cheques_deseada,
                ':cuotas'      => $cuotas_deseadas,
                ':comentarios' => $comentarios_cliente
            ]);

            $pdo->commit();
            $ok = true;
        } catch (Exception $e) {
            $pdo->rollBack();
            $errores[] = "Error al guardar la simulación: " . $e->getMessage();
        }
    }
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
<meta charset="UTF-8">
<title>Simular formas de pago - Cabañas Santiago</title>
<meta name="viewport" content="width=device-width, initial-scale=1">
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
<style>
:root{ --accent:#ffae40; }
body{ background:#f7f7f7; font-family:"Inter",system-ui,sans-serif; }
.card-modelo img{ width:100%; height:220px; object-fit:cover; border-radius:12px 12px 0 0; }
.card-modelo{ border-radius:12px; overflow:hidden; border:1px solid #eaeaea; background:#fff; }
.btn-accent{ background:var(--accent); border:none; font-weight:600; }
</style>
</head>
<body>

<div class="container py-4">
    <a href="index.php" class="btn btn-link mb-3">&larr; Volver al inicio</a>

    <div class="row g-4">
        <!-- Columna modelo -->
        <div class="col-lg-5">
            <div class="card-modelo mb-3">
                <?php
                $img_raw = trim($modelo['imagen_portada'] ?? '');
                $img_url = '';
                if ($img_raw !== '') {
                    if (strpos($img_raw, 'uploads/') === 0) {
                        $img_url = 'admin/' . $img_raw;
                    } else {
                        $img_url = $img_raw;
                    }
                }
                ?>
                <?php if ($img_url): ?>
                    <img src="<?php echo htmlspecialchars($img_url); ?>" alt="<?php echo htmlspecialchars($modelo['nombre']); ?>">
                <?php endif; ?>
                <div class="p-3">
                    <h3 class="mb-1"><?php echo htmlspecialchars($modelo['nombre']); ?></h3>
                    <p class="mb-1">
                        <strong>Metros:</strong>
                        <?php echo $modelo['metros_cubiertos']; ?> m² cubiertos
                        <?php if ($modelo['metros_galeria'] > 0): ?>
                            + <?php echo $modelo['metros_galeria']; ?> m² galería
                        <?php endif; ?>
                    </p>
                    <p class="mb-1"><strong>Sistema base:</strong> <?php echo ucfirst($modelo['sistema_default']); ?></p>
                    <p class="mb-0">
                        <strong>Precio de referencia:</strong>
                        $<?php echo number_format($modelo['precio_lista'] ?: $modelo['precio_desde'], 0, ',', '.'); ?>
                    </p>
                </div>
            </div>

            <div class="alert alert-info">
                Esta simulación te permite indicar cómo te gustaría pagar la vivienda.
                Los montos que se toman a vehículo, terreno o cheques serán definidos
                por un asesor al armar la operación real, en base a tu propuesta.
            </div>
        </div>

        <!-- Columna formulario -->
        <div class="col-lg-7">
            <h4 class="mb-3">Completá tus datos y cómo querés pagar</h4>

            <?php if ($ok): ?>
                <div class="alert alert-success">
                    <strong>¡Listo!</strong> Tu simulación fue enviada correctamente.
                    <?php if ($nuevo_codigo): ?>
                        <br>Se generó/actualizó tu usuario en el sistema.
                        <br><strong>Código de acceso:</strong> <?php echo htmlspecialchars($nuevo_codigo); ?>
                        <br>Guardalo, junto con tu DNI, para ingresar luego al portal de clientes.
                    <?php endif; ?>
                </div>
            <?php endif; ?>

            <?php if ($errores): ?>
                <div class="alert alert-danger">
                    <ul class="mb-0">
                        <?php foreach ($errores as $e): ?>
                            <li><?php echo htmlspecialchars($e); ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            <?php endif; ?>

            <form method="post" class="row g-3">

                <h5>Datos personales</h5>
                <div class="col-md-6">
                    <label class="form-label">Nombre</label>
                    <input type="text" name="nombre" required class="form-control"
                           value="<?php echo htmlspecialchars($_POST['nombre'] ?? ''); ?>">
                </div>
                <div class="col-md-6">
                    <label class="form-label">Apellido</label>
                    <input type="text" name="apellido" required class="form-control"
                           value="<?php echo htmlspecialchars($_POST['apellido'] ?? ''); ?>">
                </div>
                <div class="col-md-4">
                    <label class="form-label">DNI</label>
                    <input type="text" name="dni" required class="form-control"
                           value="<?php echo htmlspecialchars($_POST['dni'] ?? ''); ?>">
                </div>
                <div class="col-md-4">
                    <label class="form-label">Teléfono</label>
                    <input type="text" name="telefono" required class="form-control"
                           value="<?php echo htmlspecialchars($_POST['telefono'] ?? ''); ?>">
                </div>
                <div class="col-md-4">
                    <label class="form-label">Localidad</label>
                    <input type="text" name="localidad" class="form-control"
                           value="<?php echo htmlspecialchars($_POST['localidad'] ?? ''); ?>">
                </div>
                <div class="col-md-6">
                    <label class="form-label">Email</label>
                    <input type="email" name="email" required class="form-control"
                           value="<?php echo htmlspecialchars($_POST['email'] ?? ''); ?>">
                </div>

                <hr class="mt-4">

                <h5>Cómo te gustaría pagar</h5>

                <div class="col-md-6">
                    <label class="form-label">Sistema constructivo</label>
                    <select name="sistema" class="form-select">
                        <option value="estructura" <?php echo ($modelo['sistema_default']=='estructura')?'selected':''; ?>>Estructura</option>
                        <option value="estructura_plataforma" <?php echo ($modelo['sistema_default']=='estructura_plataforma')?'selected':''; ?>>Estructura + Plataforma</option>
                        <option value="llave_en_mano" <?php echo ($modelo['sistema_default']=='llave_en_mano')?'selected':''; ?>>Llave en mano</option>
                    </select>
                </div>

                <div class="col-md-6">
                    <label class="form-label">Forma de pago deseada</label>
                    <select name="forma_venta_deseada" class="form-select" required>
                        <option value="">Seleccioná una opción…</option>
                        <option value="contado"     <?php echo (($_POST['forma_venta_deseada'] ?? '')=='contado')?'selected':''; ?>>Contado</option>
                        <option value="financiado"  <?php echo (($_POST['forma_venta_deseada'] ?? '')=='financiado')?'selected':''; ?>>Financiado</option>
                        <option value="mixto"       <?php echo (($_POST['forma_venta_deseada'] ?? '')=='mixto')?'selected':''; ?>>Mixto (contado + financiación)</option>
                    </select>
                </div>

                <div class="col-md-4">
                    <label class="form-label">Seña que podrías dar</label>
                    <input type="number" step="0.01" min="0" name="senia_deseada" class="form-control"
                           value="<?php echo htmlspecialchars($_POST['senia_deseada'] ?? '0'); ?>">
                </div>

                <div class="col-md-4">
                    <label class="form-label">Cantidad de cuotas</label>
                    <input type="number" min="0" name="cuotas_deseadas" class="form-control"
                           value="<?php echo htmlspecialchars($_POST['cuotas_deseadas'] ?? '0'); ?>">
                </div>

                <div class="col-12">
                    <hr class="mt-4">
                    <h6 class="mb-2">Entregas como parte de pago (describí si corresponde)</h6>
                </div>

                <div class="col-12">
                    <label class="form-label">Vehículo a entregar (si tenés uno)</label>
                    <textarea name="entrega_vehiculo_detalle" class="form-control" rows="2"
                              placeholder="Ej: Auto Gol 2015, buen estado, papeles al día"><?php
                        echo htmlspecialchars($_POST['entrega_vehiculo_detalle'] ?? '');
                    ?></textarea>
                </div>

                <div class="col-12">
                    <label class="form-label">Terreno a entregar (si aplica)</label>
                    <textarea name="entrega_terreno_detalle" class="form-control" rows="2"
                              placeholder="Ubicación, medidas aproximadas, etc."><?php
                        echo htmlspecialchars($_POST['entrega_terreno_detalle'] ?? '');
                    ?></textarea>
                </div>

                <div class="col-12">
                    <label class="form-label">Cheques que podrías entregar</label>
                    <textarea name="entrega_cheques_detalle" class="form-control" rows="2"
                              placeholder="Cantidad de cheques, plazos, bancos, etc."><?php
                        echo htmlspecialchars($_POST['entrega_cheques_detalle'] ?? '');
                    ?></textarea>
                </div>

                <div class="col-12">
                    <label class="form-label">Comentarios u observaciones</label>
                    <textarea name="observaciones" class="form-control" rows="3"><?php
                        echo htmlspecialchars($_POST['observaciones'] ?? '');
                    ?></textarea>
                </div>

                <div class="col-12 text-end">
                    <button type="submit" class="btn btn-accent px-4">Enviar simulación</button>
                </div>

            </form>
        </div>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
