<?php
// cliente/registro.php
@session_start();
require_once '../db.php';
$pdo = pdo();

$errores = [];
$ok = false;

$nombre    = '';
$apellido  = '';
$dni       = '';
$email     = '';
$telefono  = '';
$localidad = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $nombre    = trim($_POST['nombre'] ?? '');
    $apellido  = trim($_POST['apellido'] ?? '');
    $dni       = trim($_POST['dni'] ?? '');
    $email     = trim($_POST['email'] ?? '');
    $telefono  = trim($_POST['telefono'] ?? '');
    $localidad = trim($_POST['localidad'] ?? '');
    $pass1     = trim($_POST['password'] ?? '');
    $pass2     = trim($_POST['password2'] ?? '');

    if ($nombre === '')    $errores[] = 'El nombre es obligatorio.';
    if ($apellido === '')  $errores[] = 'El apellido es obligatorio.';
    if ($dni === '')       $errores[] = 'El DNI es obligatorio.';
    if ($email === '')     $errores[] = 'El email es obligatorio.';
    if ($localidad === '') $errores[] = 'La localidad es obligatoria.';

    if ($pass1 === '' || $pass2 === '') {
        $errores[] = 'Debés ingresar la contraseña dos veces.';
    } elseif ($pass1 !== $pass2) {
        $errores[] = 'Las contraseñas no coinciden.';
    }

    if (empty($errores)) {
        try {
            // ¿Ya existe cliente con este DNI?
            $stmt = $pdo->prepare("SELECT * FROM clientes WHERE dni = :dni LIMIT 1");
            $stmt->execute([':dni' => $dni]);
            $cli = $stmt->fetch();

            $password_hash = password_hash($pass1, PASSWORD_DEFAULT);

            if ($cli) {
                // Cliente ya existe
                if (!empty($cli['password_hash'])) {
                    // Ya tiene cuenta de portal
                    $errores[] = 'Ya existe una cuenta asociada a este DNI. Ingresá por el login.';
                } else {
                    // Asegurar que tenga código de acceso
                    $codigo_acceso = $cli['codigo_acceso'];
                    if (empty($codigo_acceso)) {
                        $codigo_acceso = 'CS-' . str_pad($cli['id'], 6, '0', STR_PAD_LEFT);
                    }

                    // Actualizar datos + password + fecha_registro_portal
                    $stmt = $pdo->prepare("
                        UPDATE clientes
                        SET nombre = :nombre,
                            apellido = :apellido,
                            email = :email,
                            telefono = :tel,
                            localidad = :localidad,
                            codigo_acceso = :codigo,
                            password_hash = :pass,
                            fecha_registro_portal = NOW()
                        WHERE id = :id
                    ");
                    $stmt->execute([
                        ':nombre'    => $nombre,
                        ':apellido'  => $apellido,
                        ':email'     => $email,
                        ':tel'       => $telefono ?: null,
                        ':localidad' => $localidad,
                        ':codigo'    => $codigo_acceso,
                        ':pass'      => $password_hash,
                        ':id'        => $cli['id'],
                    ]);

                    // Refrescar cliente
                    $stmt = $pdo->prepare("SELECT * FROM clientes WHERE id = :id LIMIT 1");
                    $stmt->execute([':id' => $cli['id']]);
                    $cli = $stmt->fetch();

                    $ok = true;
                }
            } else {
                // No existía el cliente: lo creamos
                $stmt = $pdo->prepare("
                    INSERT INTO clientes
                      (nombre, apellido, dni, email, telefono, localidad,
                       estado, created_at, fecha_registro_portal, password_hash)
                    VALUES
                      (:nombre, :apellido, :dni, :email, :tel, :localidad,
                       'activo', NOW(), NOW(), :pass)
                ");
                $stmt->execute([
                    ':nombre'    => $nombre,
                    ':apellido'  => $apellido,
                    ':dni'       => $dni,
                    ':email'     => $email,
                    ':tel'       => $telefono ?: null,
                    ':localidad' => $localidad,
                    ':pass'      => $password_hash,
                ]);
                $nuevo_id = (int)$pdo->lastInsertId();

                // Generar código de acceso tipo CS-000123
                $codigo_acceso = 'CS-' . str_pad($nuevo_id, 6, '0', STR_PAD_LEFT);
                $pdo->prepare("
                    UPDATE clientes
                    SET codigo_acceso = :codigo
                    WHERE id = :id
                ")->execute([
                    ':codigo' => $codigo_acceso,
                    ':id'     => $nuevo_id,
                ]);

                // Traer cliente recién creado
                $stmt = $pdo->prepare("SELECT * FROM clientes WHERE id = :id LIMIT 1");
                $stmt->execute([':id' => $nuevo_id]);
                $cli = $stmt->fetch();

                $ok = true;
            }

            if ($ok && $cli) {
                // Logueamos al cliente
                $_SESSION['cliente_id']       = $cli['id'];
                $_SESSION['cliente_nombre']   = $cli['nombre'];
                $_SESSION['cliente_apellido'] = $cli['apellido'];
                $_SESSION['cliente_dni']      = $cli['dni'];

                // Lo mandamos al panel
                header('Location: panel.php?bienvenido=1');
                exit;
            }

        } catch (PDOException $e) {
            $errores[] = 'Error al registrar la cuenta: ' . $e->getMessage();
        }
    }
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <title>Registro de cliente - Cabañas Santiago</title>
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
</head>
<body class="bg-light d-flex align-items-center" style="min-height:100vh;">
<div class="container">
    <div class="row justify-content-center">
        <div class="col-md-6 col-lg-5">
            <div class="card shadow-sm">
                <div class="card-body">
                    <h4 class="mb-1 text-center">Crear cuenta de cliente</h4>
                    <p class="text-muted text-center mb-3" style="font-size:0.9rem;">
                        Portal de clientes de Cabañas Santiago
                    </p>

                    <?php if (!empty($errores)): ?>
                        <div class="alert alert-danger">
                            <ul class="mb-0">
                                <?php foreach ($errores as $e): ?>
                                    <li><?php echo htmlspecialchars($e); ?></li>
                                <?php endforeach; ?>
                            </ul>
                        </div>
                    <?php endif; ?>

                    <form method="post" class="row g-2">
                        <div class="col-md-6">
                            <label class="form-label">Nombre *</label>
                            <input type="text" name="nombre" class="form-control"
                                   value="<?php echo htmlspecialchars($nombre); ?>" required>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Apellido *</label>
                            <input type="text" name="apellido" class="form-control"
                                   value="<?php echo htmlspecialchars($apellido); ?>" required>
                        </div>

                        <div class="col-md-6">
                            <label class="form-label">DNI *</label>
                            <input type="text" name="dni" class="form-control"
                                   value="<?php echo htmlspecialchars($dni); ?>" required>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Localidad *</label>
                            <input type="text" name="localidad" class="form-control"
                                   value="<?php echo htmlspecialchars($localidad); ?>" required>
                        </div>

                        <div class="col-md-6">
                            <label class="form-label">Email *</label>
                            <input type="email" name="email" class="form-control"
                                   value="<?php echo htmlspecialchars($email); ?>" required>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Teléfono</label>
                            <input type="text" name="telefono" class="form-control"
                                   value="<?php echo htmlspecialchars($telefono); ?>">
                        </div>

                        <div class="col-md-6">
                            <label class="form-label">Contraseña *</label>
                            <input type="password" name="password" class="form-control" required>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Repetir contraseña *</label>
                            <input type="password" name="password2" class="form-control" required>
                        </div>

                        <div class="col-12 d-flex justify-content-between align-items-center mt-3">
                            <a href="login.php" class="btn btn-link p-0">Ya tengo cuenta</a>
                            <button class="btn btn-primary" type="submit">Crear cuenta</button>
                        </div>
                    </form>

                    <p class="text-muted mt-3 mb-0" style="font-size:0.8rem;">
                        Una vez creada la cuenta, vas a poder ingresar con tu DNI o con un código de acceso
                        que se te asigna automáticamente.
                    </p>
                </div>
            </div>
        </div>
    </div>
</div>
</body>
</html>
