<?php
// admin/pagos.php
@session_start();
if (empty($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

require_once '../db.php';
$pdo = pdo();

function h($v){ return htmlspecialchars((string)$v, ENT_QUOTES, 'UTF-8'); }

$operacion_id = isset($_GET['operacion_id']) ? (int)$_GET['operacion_id'] : 0;
if ($operacion_id <= 0) {
    die('Falta operacion_id.');
}

/* =========================
   1) Traer operación + cliente
   ========================= */
$stmt = $pdo->prepare("
    SELECT 
        o.*,
        c.nombre,
        c.apellido,
        c.dni
    FROM operaciones o
    INNER JOIN clientes c ON c.id = o.cliente_id
    WHERE o.id = :id
    LIMIT 1
");
$stmt->execute([':id' => $operacion_id]);
$op = $stmt->fetch();

if (!$op) {
    die('Operación no encontrada.');
}

/* =========================
   1.1) Calcular saldo a financiar base
   ========================= */
$precio_lista = (float)$op['precio_lista'];

$entregas_iniciales =
    (float)$op['senia'] +
    (float)$op['entrega_vehiculo'] +
    (float)$op['entrega_terreno'] +
    (float)$op['entrega_cheques'];

$saldo_financiar_base = $precio_lista - $entregas_iniciales;
if ($saldo_financiar_base < 0) $saldo_financiar_base = 0;

// Si en la tabla operaciones no está seteado o es 0, usamos el base
$saldo_financiar = (float)$op['saldo_financiar'];
if ($saldo_financiar <= 0) {
    $saldo_financiar = $saldo_financiar_base;
}

/* =========================
   1.2) Traer plan de cuotas (si existe)
   ========================= */
$stmtPlan = $pdo->prepare("
    SELECT *
    FROM operaciones_cuotas
    WHERE operacion_id = :op
    ORDER BY numero ASC
");
$stmtPlan->execute([':op' => $operacion_id]);
$plan_rows   = $stmtPlan->fetchAll(PDO::FETCH_ASSOC);
$plan_existe = !empty($plan_rows);

$errores_plan  = [];
$errores_pago  = [];
$errores_solic = [];

/* =========================
   2) Procesar POST (plan, pago o confirmación)
   ========================= */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    $accion = $_POST['accion'] ?? 'pago';

    /* --------- 2.A CREAR PLAN --------- */
    if ($accion === 'crear_plan') {

        $cant_cuotas = (int)($_POST['cant_cuotas'] ?? 0);
        if ($saldo_financiar <= 0) {
            $errores_plan[] = 'La operación no tiene saldo a financiar (revisá entregas y precio).';
        }
        if ($cant_cuotas <= 0) {
            $errores_plan[] = 'La cantidad de cuotas debe ser mayor a 0.';
        }

        if (empty($errores_plan)) {
            try {
                $pdo->beginTransaction();

                // Borramos plan previo si lo hubiera
                $pdo->prepare("DELETE FROM operaciones_cuotas WHERE operacion_id = :op")
                    ->execute([':op' => $operacion_id]);

                // Cuota base y ajuste en la última por redondeo
                $monto_base = floor(($saldo_financiar * 100) / $cant_cuotas) / 100;
                $resto      = $saldo_financiar - ($monto_base * $cant_cuotas);

                $ins = $pdo->prepare("
                    INSERT INTO operaciones_cuotas (operacion_id, numero, monto)
                    VALUES (:op, :num, :monto)
                ");

                for ($n = 1; $n <= $cant_cuotas; $n++) {
                    $monto = $monto_base;
                    if ($n == $cant_cuotas) {
                        $monto += $resto; // Ajuste centavos en la última
                    }
                    $ins->execute([
                        ':op'    => $operacion_id,
                        ':num'   => $n,
                        ':monto' => $monto
                    ]);
                }

                // Actualizo operación con info de cuotas
                $cuota_prom = round($saldo_financiar / $cant_cuotas, 2);

                $updOp = $pdo->prepare("
                    UPDATE operaciones
                    SET saldo_financiar = :saldo,
                        cuotas_cantidad = :cant,
                        cuota_monto = :monto
                    WHERE id = :id
                ");
                $updOp->execute([
                    ':saldo' => $saldo_financiar,
                    ':cant'  => $cant_cuotas,
                    ':monto' => $cuota_prom,
                    ':id'    => $operacion_id
                ]);

                $pdo->commit();
                header('Location: pagos.php?operacion_id=' . $operacion_id);
                exit;

            } catch (Exception $e) {
                if ($pdo->inTransaction()) $pdo->rollBack();
                $errores_plan[] = 'Error al crear el plan: ' . $e->getMessage();
            }
        }

    /* --------- 2.B GUARDAR CAMBIOS EN PLAN --------- */
    } elseif ($accion === 'guardar_plan') {

        $montos = $_POST['montos'] ?? [];
        if (empty($montos)) {
            $errores_plan[] = 'No se recibieron montos de cuotas.';
        }

        // Validar que todos sean > 0
        $total_nuevo = 0;
        $cant_cuotas = 0;
        foreach ($montos as $num => $valor) {
            $m = (float)$valor;
            if ($m <= 0) {
                $errores_plan[] = "El monto de la cuota $num debe ser mayor a 0.";
            }
            $montos[$num] = $m;
            $total_nuevo += $m;
            $cant_cuotas++;
        }

        if (empty($errores_plan)) {
            try {
                $pdo->beginTransaction();

                $upd = $pdo->prepare("
                    UPDATE operaciones_cuotas
                    SET monto = :monto
                    WHERE operacion_id = :op AND numero = :num
                ");

                foreach ($montos as $num => $m) {
                    $upd->execute([
                        ':monto' => $m,
                        ':op'    => $operacion_id,
                        ':num'   => (int)$num
                    ]);

                    // Reajustar estado según pagado actual
                    $pdo->prepare("
                        UPDATE operaciones_cuotas
                        SET estado = CASE
                            WHEN pagado >= monto - 0.01 THEN 'pagada'
                            WHEN pagado > 0 THEN 'parcial'
                            ELSE 'pendiente'
                        END
                        WHERE operacion_id = :op AND numero = :num
                    ")->execute([
                        ':op'  => $operacion_id,
                        ':num' => (int)$num
                    ]);
                }

                // Actualizar operación con nuevo saldo / promedio
                $cuota_prom = $cant_cuotas > 0 ? round($total_nuevo / $cant_cuotas, 2) : 0;

                $updOp = $pdo->prepare("
                    UPDATE operaciones
                    SET saldo_financiar = :saldo,
                        cuotas_cantidad = :cant,
                        cuota_monto = :monto
                    WHERE id = :id
                ");
                $updOp->execute([
                    ':saldo' => $total_nuevo,
                    ':cant'  => $cant_cuotas,
                    ':monto' => $cuota_prom,
                    ':id'    => $operacion_id
                ]);

                $pdo->commit();
                header('Location: pagos.php?operacion_id=' . $operacion_id);
                exit;

            } catch (Exception $e) {
                if ($pdo->inTransaction()) $pdo->rollBack();
                $errores_plan[] = 'Error al guardar el plan: ' . $e->getMessage();
            }
        }

    /* --------- 2.C APROBAR PAGO SOLICITADO (cliente) --------- */
    } elseif ($accion === 'aprobar_solicitado') {

        $solicitud_id = isset($_POST['solicitud_id']) ? (int)$_POST['solicitud_id'] : 0;
        if ($solicitud_id <= 0) {
            $errores_solic[] = 'Solicitud no válida.';
        } else {
            try {
                $pdo->beginTransaction();

                // Traer la solicitud
                $stmtSol = $pdo->prepare("
                    SELECT *
                    FROM pagos_solicitados
                    WHERE id = :id AND operacion_id = :op
                    LIMIT 1
                ");
                $stmtSol->execute([
                    ':id' => $solicitud_id,
                    ':op' => $operacion_id
                ]);
                $sol = $stmtSol->fetch(PDO::FETCH_ASSOC);

                if (!$sol) {
                    throw new Exception('No se encontró la solicitud de pago.');
                }
                if ($sol['estado'] !== 'pendiente') {
                    throw new Exception('La solicitud ya fue procesada.');
                }

                $monto      = (float)$sol['monto'];
                $medio      = $sol['medio'];
                $cuota_num  = !empty($sol['cuota_numero']) ? (int)$sol['cuota_numero'] : null;
                $comentario = $sol['comentario'] ?? '';

                // Definir si es pago de cuota o extra
                $tipo_pago = 'extra';
                if ($plan_existe && $cuota_num !== null && $cuota_num > 0) {
                    $tipo_pago = 'cuota';
                } else {
                    $cuota_num = null;
                }

                // Insert en pagos (pago confirmado)
                $sql = "
                    INSERT INTO pagos
                        (operacion_id, fecha_pago, monto, tipo, cuota_numero, medio,
                         referencia, observaciones, registrado_por, created_at)
                    VALUES
                        (:op_id, NOW(), :monto, :tipo, :cuota_num, :medio,
                         :ref, :obs, :user_id, NOW())
                ";
                $ref = 'Pago web ID '.$solicitud_id;
                $obs = trim('Confirmado por admin. '.$comentario);

                $stmtIns = $pdo->prepare($sql);
                $stmtIns->execute([
                    ':op_id'     => $operacion_id,
                    ':monto'     => $monto,
                    ':tipo'      => $tipo_pago,
                    ':cuota_num' => $cuota_num,
                    ':medio'     => $medio,
                    ':ref'       => $ref,
                    ':obs'       => $obs !== '' ? $obs : null,
                    ':user_id'   => (int)$_SESSION['admin_id'],
                ]);

                // Si es cuota, actualizar operaciones_cuotas
                if ($tipo_pago === 'cuota' && $cuota_num !== null) {
                    $pdo->prepare("
                        UPDATE operaciones_cuotas
                        SET pagado = pagado + :monto
                        WHERE operacion_id = :op AND numero = :num
                    ")->execute([
                        ':monto' => $monto,
                        ':op'    => $operacion_id,
                        ':num'   => $cuota_num
                    ]);

                    $pdo->prepare("
                        UPDATE operaciones_cuotas
                        SET estado = CASE
                            WHEN pagado >= monto - 0.01 THEN 'pagada'
                            WHEN pagado > 0 THEN 'parcial'
                            ELSE 'pendiente'
                        END
                        WHERE operacion_id = :op AND numero = :num
                    ")->execute([
                        ':op'  => $operacion_id,
                        ':num' => $cuota_num
                    ]);
                }

                // Marcar solicitud como aprobada
                $pdo->prepare("
                    UPDATE pagos_solicitados
                    SET estado = 'aprobado'
                    WHERE id = :id
                ")->execute([':id' => $solicitud_id]);

                $pdo->commit();
                header('Location: pagos.php?operacion_id=' . $operacion_id);
                exit;

            } catch (Exception $e) {
                if ($pdo->inTransaction()) $pdo->rollBack();
                $errores_solic[] = 'Error al aprobar el pago: ' . $e->getMessage();
            }
        }

    /* --------- 2.D RECHAZAR PAGO SOLICITADO (cliente) --------- */
    } elseif ($accion === 'rechazar_solicitado') {

        $solicitud_id = isset($_POST['solicitud_id']) ? (int)$_POST['solicitud_id'] : 0;
        if ($solicitud_id <= 0) {
            $errores_solic[] = 'Solicitud no válida.';
        } else {
            try {
                $stmtSol = $pdo->prepare("
                    UPDATE pagos_solicitados
                    SET estado = 'rechazado'
                    WHERE id = :id AND operacion_id = :op AND estado = 'pendiente'
                ");
                $stmtSol->execute([
                    ':id' => $solicitud_id,
                    ':op' => $operacion_id
                ]);

                header('Location: pagos.php?operacion_id=' . $operacion_id);
                exit;

            } catch (Exception $e) {
                $errores_solic[] = 'Error al rechazar el pago: ' . $e->getMessage();
            }
        }

    /* --------- 2.E REGISTRAR PAGO MANUAL (local) --------- */
    } else { // accion === 'pago'

        $tipo        = $_POST['tipo'] ?? 'cuota'; // 'cuota' o 'extra'
        $monto       = (float)($_POST['monto'] ?? 0);
        $medio       = $_POST['medio'] ?? '';
        $fecha_pago  = trim($_POST['fecha_pago'] ?? '');
        $referencia  = trim($_POST['referencia'] ?? '');
        $observacion = trim($_POST['observaciones'] ?? '');
        $cuota_num   = isset($_POST['cuota_numero']) && $_POST['cuota_numero'] !== ''
                        ? (int)$_POST['cuota_numero'] : null;

        // Validaciones
        if ($monto <= 0) {
            $errores_pago[] = 'El monto debe ser mayor a 0.';
        }

        $medios_validos = [
            'efectivo',
            'transferencia',
            'debito',
            'credito',
            'mercadopago',
            'pago_facil',
            'rapipago',
            'cheque',
            'otro'
        ];
        if (!in_array($medio, $medios_validos, true)) {
            $errores_pago[] = 'Medio de pago inválido.';
        }

        if ($tipo === 'cuota') {
            if (!$plan_existe) {
                $errores_pago[] = 'La operación no tiene plan de cuotas definido.';
            }
            if ($cuota_num === null) {
                $errores_pago[] = 'Debés elegir una cuota.';
            }
        } else {
            $cuota_num = null;
        }

        if (empty($errores_pago)) {
            try {
                $pdo->beginTransaction();

                // Insert pago
                if ($fecha_pago === '') {
                    $sql = "
                        INSERT INTO pagos
                            (operacion_id, fecha_pago, monto, tipo, cuota_numero, medio,
                             referencia, observaciones, registrado_por, created_at)
                        VALUES
                            (:op_id, NOW(), :monto, :tipo, :cuota_num, :medio,
                             :ref, :obs, :user_id, NOW())
                    ";
                    $params = [
                        ':op_id'     => $operacion_id,
                        ':monto'     => $monto,
                        ':tipo'      => $tipo,
                        ':cuota_num' => $cuota_num,
                        ':medio'     => $medio,
                        ':ref'       => $referencia ?: null,
                        ':obs'       => $observacion ?: null,
                        ':user_id'   => (int)$_SESSION['admin_id'],
                    ];
                } else {
                    $sql = "
                        INSERT INTO pagos
                            (operacion_id, fecha_pago, monto, tipo, cuota_numero, medio,
                             referencia, observaciones, registrado_por, created_at)
                        VALUES
                            (:op_id, :fecha_pago, :monto, :tipo, :cuota_num, :medio,
                             :ref, :obs, :user_id, NOW())
                    ";
                    $params = [
                        ':op_id'      => $operacion_id,
                        ':fecha_pago' => $fecha_pago . ' 00:00:00',
                        ':monto'      => $monto,
                        ':tipo'       => $tipo,
                        ':cuota_num'  => $cuota_num,
                        ':medio'      => $medio,
                        ':ref'        => $referencia ?: null,
                        ':obs'        => $observacion ?: null,
                        ':user_id'    => (int)$_SESSION['admin_id'],
                    ];
                }

                $stmtIns = $pdo->prepare($sql);
                $stmtIns->execute($params);

                // Si es pago de cuota, actualizar tabla operaciones_cuotas
                if ($tipo === 'cuota' && $cuota_num !== null) {
                    // sumar pagado
                    $pdo->prepare("
                        UPDATE operaciones_cuotas
                        SET pagado = pagado + :monto
                        WHERE operacion_id = :op AND numero = :num
                    ")->execute([
                        ':monto' => $monto,
                        ':op'    => $operacion_id,
                        ':num'   => $cuota_num
                    ]);

                    // ajustar estado
                    $pdo->prepare("
                        UPDATE operaciones_cuotas
                        SET estado = CASE
                            WHEN pagado >= monto - 0.01 THEN 'pagada'
                            WHEN pagado > 0 THEN 'parcial'
                            ELSE 'pendiente'
                        END
                        WHERE operacion_id = :op AND numero = :num
                    ")->execute([
                        ':op'  => $operacion_id,
                        ':num' => $cuota_num
                    ]);
                }

                $pdo->commit();
                header('Location: pagos.php?operacion_id=' . $operacion_id);
                exit;

            } catch (Exception $e) {
                if ($pdo->inTransaction()) $pdo->rollBack();
                $errores_pago[] = 'Error al registrar el pago: ' . $e->getMessage();
            }
        }
    }
}

/* =========================
   3) Recargar plan y pagos desde DB
   ========================= */
$stmtPlan->execute([':op' => $operacion_id]);
$plan_rows   = $stmtPlan->fetchAll(PDO::FETCH_ASSOC);
$plan_existe = !empty($plan_rows);

$stmt = $pdo->prepare("
    SELECT p.*, ua.usuario AS usuario_admin
    FROM pagos p
    LEFT JOIN usuarios_admin ua ON ua.id = p.registrado_por
    WHERE p.operacion_id = :op_id
    ORDER BY p.fecha_pago ASC, p.id ASC
");
$stmt->execute([':op_id' => $operacion_id]);
$pagos = $stmt->fetchAll();

/* =========================
   3.1) Traer pagos solicitados por cliente
   ========================= */
$stmt = $pdo->prepare("
    SELECT ps.*, c.nombre, c.apellido
    FROM pagos_solicitados ps
    INNER JOIN clientes c ON c.id = ps.cliente_id
    WHERE ps.operacion_id = :op
    ORDER BY ps.fecha_solicitud DESC
");
$stmt->execute([':op' => $operacion_id]);
$pagos_solic = $stmt->fetchAll();

/* =========================
   4) Totales y estado de cuotas
   ========================= */
// Totales pagos
$total_pagos_cuotas = 0;
$total_pagos_extra  = 0;

foreach ($pagos as $p) {
    if ($p['tipo'] === 'cuota') {
        $total_pagos_cuotas += (float)$p['monto'];
    } else {
        $total_pagos_extra  += (float)$p['monto'];
    }
}
$total_pagos = $total_pagos_cuotas + $total_pagos_extra;

// Si ya tenemos plan, saldo_financiar lo tomamos como suma de montos
if ($plan_existe) {
    $saldo_financiar = 0;
    foreach ($plan_rows as $r) {
        $saldo_financiar += (float)$r['monto'];
    }
}

// Saldo pendiente
$saldo_pendiente = $saldo_financiar - $total_pagos_cuotas - $total_pagos_extra;
if ($saldo_pendiente < 0) $saldo_pendiente = 0;

// Armar array de cuotas para mostrar
$cuotas        = [];
$proxima_cuota = null;

if ($plan_existe) {
    foreach ($plan_rows as $r) {
        $n = (int)$r['numero'];
        $cuotas[$n] = [
            'numero' => $n,
            'monto'  => (float)$r['monto'],
            'pagado' => (float)$r['pagado'],
            'estado' => $r['estado']
        ];
    }

    // próxima cuota pendiente
    foreach ($cuotas as $n => $info) {
        if ($info['estado'] !== 'pagada') {
            $proxima_cuota = $n;
            break;
        }
    }
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <title>Pagos de la operación <?php echo h($op['codigo_operacion']); ?> - Cabañas Santiago</title>
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
</head>
<body class="bg-light">
<nav class="navbar navbar-expand-lg navbar-dark bg-primary">
    <div class="container-fluid">
        <a class="navbar-brand" href="panel.php">Cabañas Santiago - Admin</a>
        <div class="d-flex">
            <a class="btn btn-outline-light btn-sm me-2" href="clientes.php">Clientes</a>
            <a class="btn btn-outline-light btn-sm" href="logout.php">Salir</a>
        </div>
    </div>
</nav>

<div class="container py-4">
    <div class="mb-3">
        <h3 class="mb-1">Pagos de la operación</h3>
        <p class="mb-0">
            <strong>Cliente:</strong>
            <?php echo h($op['apellido'] . ', ' . $op['nombre']); ?>
            (DNI: <?php echo h($op['dni']); ?>)
        </p>
        <p class="mb-0">
            <strong>Código operación:</strong> <?php echo h($op['codigo_operacion']); ?><br>
            <strong>Modelo:</strong> <?php echo h($op['modelo']); ?><br>
            <strong>Sistema:</strong> <?php echo h($op['sistema']); ?> |
            <strong>Forma de venta:</strong> <?php echo h($op['forma_venta']); ?>
        </p>
    </div>

    <!-- RESUMEN -->
    <div class="row g-3 mb-4">
        <div class="col-md-3">
            <div class="card shadow-sm h-100">
                <div class="card-body">
                    <h6 class="text-muted">Precio de lista</h6>
                    <p class="h5 mb-0">
                        $ <?php echo number_format($precio_lista, 2, ',', '.'); ?>
                    </p>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card shadow-sm h-100">
                <div class="card-body">
                    <h6 class="text-muted">Entregas iniciales</h6>
                    <p class="h5 mb-0">
                        $ <?php echo number_format($entregas_iniciales, 2, ',', '.'); ?>
                    </p>
                    <small class="text-muted">
                        Seña + vehículo + terreno + cheques
                    </small>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card shadow-sm h-100">
                <div class="card-body">
                    <h6 class="text-muted">Saldo a financiar (plan)</h6>
                    <p class="h5 mb-0">
                        $ <?php echo number_format($saldo_financiar, 2, ',', '.'); ?>
                    </p>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card shadow-sm h-100 border-success">
                <div class="card-body">
                    <h6 class="text-muted">Saldo pendiente</h6>
                    <p class="h4 mb-0 text-success">
                        $ <?php echo number_format($saldo_pendiente, 2, ',', '.'); ?>
                    </p>
                </div>
            </div>
        </div>
    </div>

    <!-- ERRORES DE SOLICITUDES -->
    <?php if (!empty($errores_solic)): ?>
        <div class="alert alert-danger">
            <ul class="mb-0">
                <?php foreach ($errores_solic as $e): ?>
                    <li><?php echo h($e); ?></li>
                <?php endforeach; ?>
            </ul>
        </div>
    <?php endif; ?>

    <!-- PAGOS CON COMPROBANTE DEL CLIENTE -->
    <div class="card mb-4 shadow-sm">
        <div class="card-body">
            <h5 class="card-title mb-3">Pagos cargados por el cliente (comprobantes)</h5>

            <?php if (empty($pagos_solic)): ?>
                <div class="alert alert-info mb-0">
                    No hay pagos con comprobante enviados por el cliente para esta operación.
                </div>
            <?php else: ?>
                <div class="table-responsive">
                    <table class="table table-sm table-striped align-middle">
                        <thead class="table-light">
                        <tr>
                            <th>Fecha envío</th>
                            <th>Cliente</th>
                            <th>Cuota</th>
                            <th>Monto</th>
                            <th>Medio</th>
                            <th>Comprobante</th>
                            <th>Comentario</th>
                            <th>Estado</th>
                            <th style="width: 150px;">Acciones</th>
                        </tr>
                        </thead>
                        <tbody>
                        <?php foreach ($pagos_solic as $ps): ?>
                            <tr>
                                <td><?php echo h($ps['fecha_solicitud']); ?></td>
                                <td><?php echo h($ps['apellido'].' '.$ps['nombre']); ?></td>
                                <td><?php echo $ps['cuota_numero'] ? (int)$ps['cuota_numero'] : '-'; ?></td>
                                <td>$ <?php echo number_format($ps['monto'], 2, ',', '.'); ?></td>
                                <td><?php echo h($ps['medio']); ?></td>
                                <td>
                                    <?php if (!empty($ps['comprobante_path'])): ?>
                                        <a href="../<?php echo h($ps['comprobante_path']); ?>" target="_blank"
                                           class="btn btn-sm btn-outline-secondary">
                                            Ver
                                        </a>
                                    <?php else: ?>
                                        -
                                    <?php endif; ?>
                                </td>
                                <td><?php echo nl2br(h($ps['comentario'] ?? '')); ?></td>
                                <td>
                                    <?php
                                    if ($ps['estado'] === 'aprobado') {
                                        echo '<span class="badge bg-success">Aprobado</span>';
                                    } elseif ($ps['estado'] === 'rechazado') {
                                        echo '<span class="badge bg-danger">Rechazado</span>';
                                    } else {
                                        echo '<span class="badge bg-warning text-dark">Pendiente</span>';
                                    }
                                    ?>
                                </td>
                                <td>
                                    <?php if ($ps['estado'] === 'pendiente'): ?>
                                        <form method="post" class="d-inline">
                                            <input type="hidden" name="accion" value="aprobar_solicitado">
                                            <input type="hidden" name="solicitud_id" value="<?php echo (int)$ps['id']; ?>">
                                            <button type="submit" class="btn btn-sm btn-success mb-1">
                                                Aprobar
                                            </button>
                                        </form>
                                        <form method="post" class="d-inline">
                                            <input type="hidden" name="accion" value="rechazar_solicitado">
                                            <input type="hidden" name="solicitud_id" value="<?php echo (int)$ps['id']; ?>">
                                            <button type="submit" class="btn btn-sm btn-outline-danger mb-1">
                                                Rechazar
                                            </button>
                                        </form>
                                    <?php else: ?>
                                        <small class="text-muted">Ya procesado</small>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- PLAN DE PAGOS -->
    <div class="card mb-4 shadow-sm">
        <div class="card-body">
            <h5 class="card-title mb-3">Plan de pagos</h5>

            <?php if (!empty($errores_plan)): ?>
                <div class="alert alert-danger">
                    <ul class="mb-0">
                        <?php foreach ($errores_plan as $e): ?>
                            <li><?php echo h($e); ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            <?php endif; ?>

            <?php if (!$plan_existe): ?>
                <!-- Crear plan -->
                <p>Saldo a financiar: <strong>$ <?php echo number_format($saldo_financiar, 2, ',', '.'); ?></strong></p>

                <form method="post" class="row g-3">
                    <input type="hidden" name="accion" value="crear_plan">
                    <div class="col-md-3">
                        <label class="form-label">Cantidad de cuotas</label>
                        <input type="number" name="cant_cuotas" class="form-control" min="1"
                               value="<?php echo (int)$op['cuotas_cantidad'] ?: 1; ?>" required>
                    </div>
                    <div class="col-12 d-flex justify-content-between mt-2">
                        <a href="operaciones.php?cliente_id=<?php echo (int)$op['cliente_id']; ?>"
                           class="btn btn-secondary">Volver a operaciones</a>
                        <button type="submit" class="btn btn-primary">Generar plan de cuotas iguales</button>
                    </div>
                </form>

            <?php else: ?>
                <!-- Editar plan existente -->
                <p class="mb-2">
                    Saldo a financiar (suma de cuotas):
                    <strong>$ <?php echo number_format($saldo_financiar, 2, ',', '.'); ?></strong><br>
                    Cantidad de cuotas: <strong><?php echo count($cuotas); ?></strong>
                </p>

                <!-- Herramienta de % -->
                <div class="row g-3 mb-2">
                    <div class="col-md-3">
                        <label class="form-label mb-1">Ajustar todas las cuotas por %</label>
                        <div class="input-group">
                            <input type="number" step="0.01" id="porcentaje" class="form-control" placeholder="Ej: 10">
                            <button type="button" class="btn btn-outline-success" onclick="aplicarPorcentaje(1)">+ %</button>
                            <button type="button" class="btn btn-outline-danger" onclick="aplicarPorcentaje(-1)">- %</button>
                        </div>
                        <small class="text-muted">
                            Afecta solo los montos visibles (no guarda hasta que grabes el plan).
                        </small>
                    </div>
                </div>

                <form method="post">
                    <input type="hidden" name="accion" value="guardar_plan">
                    <div class="table-responsive mb-3">
                        <table class="table table-sm table-striped align-middle">
                            <thead class="table-light">
                            <tr>
                                <th>Cuota</th>
                                <th>Monto</th>
                                <th>Pagado</th>
                                <th>Estado</th>
                            </tr>
                            </thead>
                            <tbody>
                            <?php foreach ($cuotas as $n => $info): ?>
                                <tr>
                                    <td><?php echo (int)$n; ?></td>
                                    <td>
                                        <div class="input-group input-group-sm">
                                            <span class="input-group-text">$</span>
                                            <input type="number" step="0.01" min="0"
                                                   class="form-control monto-cuota"
                                                   name="montos[<?php echo (int)$n; ?>]"
                                                   value="<?php echo number_format($info['monto'], 2, '.', ''); ?>">
                                        </div>
                                    </td>
                                    <td>$ <?php echo number_format($info['pagado'], 2, ',', '.'); ?></td>
                                    <td><?php echo h($info['estado']); ?></td>
                                </tr>
                            <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>

                    <div class="d-flex justify-content-between">
                        <a href="operaciones.php?cliente_id=<?php echo (int)$op['cliente_id']; ?>"
                           class="btn btn-secondary">Volver a operaciones</a>
                        <button type="submit" class="btn btn-primary">Guardar cambios del plan</button>
                    </div>
                </form>
            <?php endif; ?>
        </div>
    </div>

    <!-- REGISTRAR PAGO MANUAL -->
    <div class="card mb-4 shadow-sm">
        <div class="card-body">
            <h5 class="card-title mb-3">Registrar nuevo pago (manual)</h5>

            <?php if (!empty($errores_pago)): ?>
                <div class="alert alert-danger">
                    <ul class="mb-0">
                        <?php foreach ($errores_pago as $e): ?>
                            <li><?php echo h($e); ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            <?php endif; ?>

            <form method="post" class="row g-3">
                <input type="hidden" name="accion" value="pago">

                <div class="col-md-3">
                    <label class="form-label">Tipo de pago *</label>
                    <select name="tipo" id="tipo" class="form-select" onchange="toggleTipoPago()" required>
                        <option value="cuota">Cuota del plan</option>
                        <option value="extra">Pago extra / adelanto</option>
                    </select>
                </div>

                <div class="col-md-3" id="bloque-cuota">
                    <label class="form-label">Cuota</label>
                    <select name="cuota_numero" id="cuota_numero" class="form-select">
                        <?php if ($plan_existe): ?>
                            <option value="">Seleccionar...</option>
                            <?php foreach ($cuotas as $n => $info): ?>
                                <?php $selected = ($proxima_cuota === $n) ? 'selected' : ''; ?>
                                <option value="<?php echo (int)$n; ?>" <?php echo $selected; ?>>
                                    Cuota <?php echo (int)$n; ?> (<?php echo $info['estado']; ?>)
                                </option>
                            <?php endforeach; ?>
                        <?php else: ?>
                            <option value="">Sin plan definido</option>
                        <?php endif; ?>
                    </select>
                    <small class="text-muted">Sugerida: próxima pendiente.</small>
                </div>

                <div class="col-md-3">
                    <label class="form-label">Fecha de pago</label>
                    <input type="date" name="fecha_pago" class="form-control">
                    <small class="text-muted">Vacío = hoy.</small>
                </div>

                <div class="col-md-3">
                    <label class="form-label">Monto *</label>
                    <input type="number" step="0.01" min="0" name="monto" class="form-control" required>
                </div>

                <div class="col-md-3">
                    <label class="form-label">Medio de pago *</label>
                    <select name="medio" class="form-select" required>
                        <option value="">Seleccionar...</option>
                        <option value="efectivo">Efectivo</option>
                        <option value="transferencia">Transferencia</option>
                        <option value="debito">Débito</option>
                        <option value="credito">Crédito</option>
                        <option value="mercadopago">MercadoPago</option>
                        <option value="pago_facil">Pago Fácil</option>
                        <option value="rapipago">Rapipago</option>
                        <option value="cheque">Cheque</option>
                        <option value="otro">Otro</option>
                    </select>
                </div>

                <div class="col-md-3">
                    <label class="form-label">Referencia / comprobante</label>
                    <input type="text" name="referencia" class="form-control"
                           placeholder="Nº operación, cupón, etc.">
                </div>

                <div class="col-12">
                    <label class="form-label">Observaciones</label>
                    <textarea name="observaciones" class="form-control" rows="2"
                              placeholder="Detalle del pago (opcional)"></textarea>
                </div>

                <div class="col-12 d-flex justify-content-between mt-2">
                    <a href="operaciones.php?cliente_id=<?php echo (int)$op['cliente_id']; ?>"
                       class="btn btn-secondary">Volver a operaciones</a>
                    <button type="submit" class="btn btn-primary">Guardar pago</button>
                </div>
            </form>
        </div>
    </div>

    <!-- ESTADO DE CUOTAS -->
    <?php if ($plan_existe): ?>
        <h5 class="mb-3">Estado de cuotas</h5>
        <div class="table-responsive mb-4">
            <table class="table table-sm table-striped align-middle">
                <thead class="table-light">
                <tr>
                    <th>Cuota</th>
                    <th>Monto plan</th>
                    <th>Pagado</th>
                    <th>Estado</th>
                </tr>
                </thead>
                <tbody>
                <?php foreach ($cuotas as $info): ?>
                    <tr>
                        <td><?php echo (int)$info['numero']; ?></td>
                        <td>$ <?php echo number_format($info['monto'], 2, ',', '.'); ?></td>
                        <td>$ <?php echo number_format($info['pagado'], 2, ',', '.'); ?></td>
                        <td><?php echo h($info['estado']); ?></td>
                    </tr>
                <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    <?php endif; ?>

    <!-- HISTORIAL DE PAGOS -->
    <h5 class="mb-3">Historial de pagos confirmados</h5>

    <?php if (empty($pagos)): ?>
        <div class="alert alert-info">Todavía no hay pagos cargados para esta operación.</div>
    <?php else: ?>
        <div class="table-responsive">
            <table class="table table-sm table-striped align-middle">
                <thead class="table-light">
                <tr>
                    <th>Fecha</th>
                    <th>Tipo</th>
                    <th>Cuota</th>
                    <th>Monto</th>
                    <th>Medio</th>
                    <th>Referencia</th>
                    <th>Registrado por</th>
                    <th>Observaciones</th>
                </tr>
                </thead>
                <tbody>
                <?php foreach ($pagos as $p): ?>
                    <tr>
                        <td><?php echo h($p['fecha_pago']); ?></td>
                        <td><?php echo h($p['tipo']); ?></td>
                        <td><?php echo $p['tipo']==='cuota' ? (int)$p['cuota_numero'] : '-'; ?></td>
                        <td>$ <?php echo number_format($p['monto'], 2, ',', '.'); ?></td>
                        <td><?php echo h($p['medio']); ?></td>
                        <td><?php echo h($p['referencia'] ?? ''); ?></td>
                        <td><?php echo h($p['usuario_admin'] ?? ''); ?></td>
                        <td><?php echo nl2br(h($p['observaciones'] ?? '')); ?></td>
                    </tr>
                <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    <?php endif; ?>
</div>

<script>
function toggleTipoPago(){
    const tipo = document.getElementById('tipo').value;
    const bloqueCuota = document.getElementById('bloque-cuota');
    if(tipo === 'cuota'){
        bloqueCuota.style.display = 'block';
    } else {
        bloqueCuota.style.display = 'none';
    }
}
document.addEventListener('DOMContentLoaded', toggleTipoPago);

function aplicarPorcentaje(signo){
    const porcInput = document.getElementById('porcentaje');
    let p = parseFloat(porcInput.value);
    if(isNaN(p) || p === 0) return;
    p = p / 100 * signo;

    const inputs = document.querySelectorAll('.monto-cuota');
    inputs.forEach(function(inp){
        let val = parseFloat(inp.value.replace(',', '.'));
        if(isNaN(val)) return;
        val = val * (1 + p);
        inp.value = val.toFixed(2);
    });
}
</script>

</body>
</html>
